@extends('layouts.UserLayout')

@push('styles')
<style>
/* ================= BASE STYLES ================= */
body {
    background: #f8fafc;
    font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
    margin: 0;
    padding: 0;
}

/* ================= REVIEW CONTAINER ================= */
.review-container {
    max-width: 1200px;
    margin: 0 auto;
    padding: 20px;
}

.review-header {
    background: linear-gradient(135deg, #1e40af 0%, #3b82f6 100%);
    color: white;
    padding: 30px;
    border-radius: 16px;
    margin-bottom: 30px;
    box-shadow: 0 10px 25px rgba(59, 130, 246, 0.15);
}

.review-header h1 {
    margin: 0 0 10px 0;
    font-size: 2.2rem;
    font-weight: 800;
}

.review-header .subtitle {
    font-size: 1.1rem;
    opacity: 0.9;
    margin-bottom: 20px;
}

.review-stats {
    display: flex;
    gap: 20px;
    flex-wrap: wrap;
    margin-top: 20px;
}

.stat-box {
    background: rgba(255, 255, 255, 0.1);
    padding: 15px 20px;
    border-radius: 10px;
    backdrop-filter: blur(10px);
    min-width: 200px;
}

.stat-box .stat-label {
    font-size: 0.9rem;
    opacity: 0.8;
    margin-bottom: 5px;
}

.stat-box .stat-value {
    font-size: 1.8rem;
    font-weight: 700;
}

/* ================= ROOM NAVIGATION ================= */
.room-navigation {
    position: sticky;
    top: 0;
    background: white;
    padding: 15px;
    border-radius: 12px;
    margin-bottom: 30px;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.08);
    z-index: 100;
    border: 1px solid #e5e7eb;
}

.room-nav-title {
    font-size: 1.1rem;
    font-weight: 600;
    color: #374151;
    margin-bottom: 15px;
    display: flex;
    align-items: center;
    gap: 10px;
}

.room-nav-title i {
    color: #3b82f6;
}

.room-nav-buttons {
    display: flex;
    gap: 10px;
    flex-wrap: wrap;
}

.room-nav-btn {
    padding: 10px 20px;
    background: #f3f4f6;
    border: 2px solid transparent;
    border-radius: 8px;
    color: #374151;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.2s;
    text-decoration: none;
    display: inline-block;
}

.room-nav-btn:hover {
    background: #e5e7eb;
    transform: translateY(-2px);
}

.room-nav-btn.active {
    background: #3b82f6;
    color: white;
    border-color: #2563eb;
    box-shadow: 0 4px 8px rgba(59, 130, 246, 0.3);
}

.room-nav-btn.completed {
    background: #10b981;
    color: white;
}

.room-nav-btn.completed:hover {
    background: #059669;
}

/* ================= ROOM SECTION ================= */
.room-section {
    background: white;
    border-radius: 16px;
    padding: 0;
    margin-bottom: 40px;
    box-shadow: 0 6px 20px rgba(0, 0, 0, 0.08);
    border: 1px solid #e5e7eb;
    overflow: hidden;
}

.room-header {
    background: linear-gradient(to right, #4f46e5, #3b82f6);
    color: white;
    padding: 25px 30px;
    position: relative;
}

.room-header h2 {
    margin: 0;
    font-size: 1.8rem;
    font-weight: 700;
    display: flex;
    align-items: center;
    gap: 12px;
}

.room-badge {
    background: rgba(255, 255, 255, 0.2);
    padding: 4px 12px;
    border-radius: 20px;
    font-size: 0.9rem;
    font-weight: 600;
    display: inline-flex;
    align-items: center;
    gap: 5px;
}

.room-meta {
    display: flex;
    gap: 20px;
    margin-top: 15px;
    flex-wrap: wrap;
}

.room-meta-item {
    display: flex;
    align-items: center;
    gap: 8px;
    font-size: 0.95rem;
    opacity: 0.9;
}

.room-meta-item i {
    font-size: 1.1rem;
}

/* ================= QUESTIONS SECTION ================= */
.questions-container {
    padding: 30px;
}

.section-card {
    background: #f9fafb;
    border-radius: 12px;
    padding: 25px;
    margin-bottom: 25px;
    border-left: 5px solid #3b82f6;
}

.section-card-header {
    font-size: 1.4rem;
    font-weight: 700;
    color: #1f2937;
    margin-bottom: 20px;
    padding-bottom: 15px;
    border-bottom: 2px solid #e5e7eb;
    display: flex;
    align-items: center;
    gap: 10px;
}

/* ================= QUESTION CARD ================= */
.question-card {
    background: white;
    border-radius: 10px;
    padding: 20px;
    margin-bottom: 20px;
    border: 1px solid #e5e7eb;
    transition: transform 0.2s;
}

.question-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
}

.question-text {
    font-weight: 600;
    font-size: 1.1rem;
    color: #111827;
    margin-bottom: 15px;
    line-height: 1.5;
}

.answer-display {
    display: flex;
    align-items: center;
    gap: 15px;
    margin-bottom: 15px;
}

.answer-badge {
    padding: 6px 15px;
    border-radius: 8px;
    font-weight: 600;
    font-size: 0.9rem;
    display: inline-flex;
    align-items: center;
    gap: 6px;
}

.answer-badge.yes, .answer-badge.good {
    background: #d1fae5;
    color: #065f46;
    border: 1px solid #10b981;
}

.answer-badge.no, .answer-badge.bad {
    background: #fee2e2;
    color: #991b1b;
    border: 1px solid #ef4444;
}

.comment-box {
    background: #f8fafc;
    border-radius: 8px;
    padding: 15px;
    margin-top: 15px;
    border-left: 4px solid #6b7280;
}

.comment-label {
    font-weight: 600;
    color: #4b5563;
    margin-bottom: 8px;
    font-size: 0.9rem;
}

.comment-text {
    color: #374151;
    line-height: 1.6;
    white-space: pre-line;
}

/* ================= IMAGE DISPLAY - DESKTOP ================= */
.image-container {
    max-width: 600px;
    width: 100%;
    border-radius: 12px;
    overflow: hidden;
    border: 2px solid #e5e7eb;
    box-shadow: 0 6px 14px rgba(0, 0, 0, 0.08);
    background: #f9fafb;
    display: flex;
    justify-content: center;
    align-items: center;
}

.image-container img {
    width: 100%;
    height: auto;
    display: block;
    object-fit: contain;
    transition: transform 0.3s ease;
    cursor: pointer;
}

.image-container img:hover {
    transform: scale(1.02);
}

/* Sub-question image */
.sub-question .image-container {
    max-width: 400px;
}

/* ================= SUB QUESTIONS ================= */
.sub-question-container {
    background: #f1f5f9;
    border-radius: 8px;
    padding: 15px;
    margin-top: 15px;
    border-left: 4px solid #64748b;
}

.sub-question {
    background: white;
    border-radius: 6px;
    padding: 15px;
    margin-bottom: 12px;
    border: 1px solid #e2e8f0;
}

.sub-question:last-child {
    margin-bottom: 0;
}

.sub-question-text {
    font-weight: 500;
    color: #334155;
    margin-bottom: 10px;
    font-size: 1rem;
}

/* ================= OVERALL COMMENT ================= */
.overall-comment-section {
    background: linear-gradient(to right, #f0f9ff, #e0f2fe);
    border-radius: 12px;
    padding: 25px;
    margin: 30px 0;
    border: 2px solid #0ea5e9;
}

.overall-comment-header {
    display: flex;
    align-items: center;
    gap: 12px;
    margin-bottom: 15px;
}

.overall-comment-header i {
    color: #0ea5e9;
    font-size: 1.5rem;
}

.overall-comment-header h3 {
    margin: 0;
    color: #0369a1;
    font-size: 1.3rem;
    font-weight: 700;
}

.overall-comment-text {
    background: white;
    border-radius: 8px;
    padding: 20px;
    border: 1px solid #bae6fd;
    color: #0c4a6e;
    line-height: 1.7;
    white-space: pre-line;
    font-size: 1rem;
}

/* ================= NAVIGATION BUTTONS ================= */
.review-navigation {
    display: flex;
    justify-content: space-between;
    margin-top: 40px;
    padding-top: 25px;
    border-top: 2px solid #e5e7eb;
}

.nav-btn {
    padding: 14px 30px;
    border-radius: 10px;
    border: none;
    font-weight: 600;
    cursor: pointer;
    font-size: 1rem;
    transition: all 0.2s;
    text-decoration: none;
    display: inline-flex;
    align-items: center;
    gap: 8px;
}

.nav-btn i {
    font-size: 1.1rem;
}

.btn-back {
    background: #f3f4f6;
    color: #374151;
}

.btn-back:hover {
    background: #e5e7eb;
    transform: translateX(-3px);
}

.btn-print {
    background: #10b981;
    color: white;
}

.btn-print:hover {
    background: #059669;
    transform: translateY(-2px);
}

.btn-edit {
    background: #3b82f6;
    color: white;
}

.btn-edit:hover {
    background: #2563eb;
    transform: translateY(-2px);
}

/* ================= EMPTY STATE ================= */
.empty-state {
    text-align: center;
    padding: 60px 20px;
    color: #6b7280;
}

.empty-state i {
    font-size: 4rem;
    margin-bottom: 20px;
    color: #d1d5db;
}

.empty-state h3 {
    font-size: 1.5rem;
    margin-bottom: 10px;
    color: #374151;
}

.empty-state p {
    font-size: 1rem;
    max-width: 500px;
    margin: 0 auto 30px;
    line-height: 1.6;
}

/* ================= IMAGE MODAL ================= */
.image-modal {
    display: none;
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.95);
    z-index: 10000;
    align-items: center;
    justify-content: center;
    padding: 20px;
    box-sizing: border-box;
}

.image-modal-content {
    position: relative;
    max-width: 90vw;
    max-height: 90vh;
    display: flex;
    justify-content: center;
    align-items: center;
}

.image-modal-content img {
    max-width: 100%;
    max-height: 90vh;
    width: auto;
    height: auto;
    object-fit: contain;
    border-radius: 8px;
    box-shadow: 0 8px 25px rgba(0, 0, 0, 0.5);
}

.image-modal-close {
    position: absolute;
    top: -40px;
    right: 0;
    color: white;
    font-size: 40px;
    cursor: pointer;
    background: none;
    border: none;
    transition: color 0.2s;
    z-index: 10001;
}

.image-modal-close:hover {
    color: #ef4444;
}

/* ================= DESKTOP SECTION ================= */
.desktop-section {
    max-width: 1200px;
    margin: 0 auto;
    padding: 20px;
}

/* ================= MOBILE SECTION STYLES ================= */
.mobile-section {
    padding: 16px;
    padding-bottom: 100px;
    background: #f8fafc;
    min-height: 100vh;
}

.mobile-review-header {
    padding: 20px;
    margin-bottom: 20px;
}

.mobile-review-header h1 {
    font-size: 1.6rem;
}

.mobile-stats-grid {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 12px;
    margin-top: 20px;
}

.mobile-stat-box {
    background: rgba(255, 255, 255, 0.1);
    padding: 12px;
    border-radius: 10px;
    backdrop-filter: blur(10px);
}

.mobile-stat-box .stat-label {
    font-size: 0.8rem;
    opacity: 0.8;
    margin-bottom: 4px;
}

.mobile-stat-box .stat-value {
    font-size: 1.3rem;
    font-weight: 700;
}

.mobile-room-navigation {
    margin-bottom: 20px;
    padding: 12px;
    background: white;
    border-radius: 12px;
    border: 1px solid #e5e7eb;
}

.mobile-room-nav-buttons {
    display: flex;
    gap: 8px;
    overflow-x: auto;
    padding-bottom: 4px;
    -webkit-overflow-scrolling: touch;
}

.mobile-room-nav-btn {
    padding: 8px 16px;
    background: #f3f4f6;
    border-radius: 8px;
    color: #374151;
    font-weight: 600;
    font-size: 0.85rem;
    text-decoration: none;
    white-space: nowrap;
    display: inline-block;
}

.mobile-room-nav-btn.completed {
    background: #10b981;
    color: white;
}

.mobile-room-header {
    padding: 20px;
}

.mobile-room-header h2 {
    font-size: 1.4rem;
    flex-direction: column;
    align-items: flex-start;
    gap: 8px;
}

.mobile-room-meta {
    display: flex;
    flex-direction: column;
    gap: 8px;
    margin-top: 12px;
}

.mobile-room-meta-item {
    font-size: 0.85rem;
}

.mobile-questions-container {
    padding: 16px;
}

.mobile-section-card {
    padding: 16px;
    margin-bottom: 20px;
}

.mobile-section-card-header {
    font-size: 1.2rem;
    margin-bottom: 16px;
}

.mobile-question-card {
    padding: 16px;
}

.mobile-question-text {
    font-size: 1rem;
}

.mobile-answer-badge {
    padding: 4px 12px;
    font-size: 0.8rem;
}

.mobile-comment-box {
    padding: 12px;
    margin-top: 12px;
}

.mobile-image-container {
    max-width: 100%;
}

.mobile-overall-comment-section {
    padding: 20px;
    margin: 20px 0;
}

.mobile-overall-comment-header h3 {
    font-size: 1.1rem;
}

.mobile-overall-comment-text {
    padding: 16px;
    font-size: 0.9rem;
}

.mobile-review-navigation {
    flex-direction: column;
    gap: 12px;
    margin-top: 30px;
    padding-top: 20px;
}

.mobile-nav-btn {
    width: 100%;
    justify-content: center;
    padding: 12px 20px;
    font-size: 0.95rem;
}

/* ================= MOBILE IMAGE ADJUSTMENTS ================= */
@media (max-width: 1023px) {
    .image-container {
        max-width: 100%;
        max-height: 300px;
        border-radius: 10px;
    }

    .image-container img {
        width: 100%;
        height: 100%;
        object-fit: contain;
    }

    .sub-question .image-container {
        max-width: 100%;
        max-height: 200px;
    }
    
    .image-modal-content img {
        max-height: 80vh;
    }
    
    .image-modal-close {
        top: -35px;
        font-size: 35px;
        right: 5px;
    }
}

/* ================= RESPONSIVE HIDING ================= */
@media (min-width: 1024px) {
    .mobile-section {
        display: none !important;
    }
    .desktop-section {
        display: block !important;
    }
}

@media (max-width: 1023px) {
    .desktop-section {
        display: none !important;
    }
    .mobile-section {
        display: block !important;
    }
}

/* ================= PRINT STYLES ================= */
@media print {
    .room-navigation,
    .review-navigation,
    .nav-btn,
    .mobile-room-navigation,
    .mobile-review-navigation {
        display: none !important;
    }
    
    .review-container {
        max-width: 100%;
        padding: 0;
    }
    
    .room-section {
        break-inside: avoid;
        box-shadow: none;
        border: 1px solid #000;
    }
    
    .question-card {
        break-inside: avoid;
    }
    
    body {
        background: white !important;
    }
}
</style>
@endpush

{{-- ================= DESKTOP SECTION ================= --}}
@section('content')
@php
    // Default values to prevent undefined variable errors
    $checklist = $checklist ?? null;
    $submissions = $submissions ?? [];
    $sections = $sections ?? collect();
    $allQuestionAnswers = $allQuestionAnswers ?? [];
    $allSubQuestionAnswers = $allSubQuestionAnswers ?? [];
@endphp

@if(!$checklist)
    <div class="desktop-section">
        <div class="review-container" style="padding: 20px;">
            <div class="alert alert-danger" style="padding: 20px; background: #fee2e2; color: #991b1b; border-radius: 10px;">
                <h3 style="margin-top: 0;"><i class="fas fa-exclamation-triangle"></i> Checklist Not Found</h3>
                <p>The checklist you're trying to review does not exist or has been deleted.</p>
                <a href="{{ route('user.dashboard') }}" class="nav-btn btn-back">
                    <i class="fas fa-arrow-left"></i> Back to Dashboard
                </a>
            </div>
        </div>
    </div>
@else
<div class="desktop-section">
    <div class="review-container">
        {{-- Header --}}
        <div class="review-header">
            <h1>
                <i class="fas fa-clipboard-check"></i>
                {{ $checklist->hotel }} - {{ $checklist->title }} Review
            </h1>
            <div class="subtitle">
                Summary of all completed inspections for {{ now()->format('F j, Y') }}
            </div>
            
            <div class="review-stats">
                <div class="stat-box">
                    <div class="stat-label">Total {{ $checklist->mod_type === 'room_inspection' ? 'Rooms' : 'Checklists' }} Inspected</div>
                    <div class="stat-value">
                        @if($checklist->mod_type === 'room_inspection')
                            {{ count($submissions) }}
                        @else
                            {{ count($submissions) > 0 ? 1 : 0 }}
                        @endif
                    </div>
                </div>
                <div class="stat-box">
                    <div class="stat-label">Hotel</div>
                    <div class="stat-value">{{ $checklist->hotel }}</div>
                </div>
                <div class="stat-box">
                    <div class="stat-label">Inspection Type</div>
                    <div class="stat-value">
                        @if($checklist->mod_type === 'room_inspection')
                            Room Inspection
                        @else
                            General Operation
                        @endif
                    </div>
                </div>
                <div class="stat-box">
                    <div class="stat-label">Completion Date</div>
                    <div class="stat-value">
                        {{ now()->format('M d, Y') }}
                    </div>
                </div>
            </div>
        </div>

        {{-- Room Navigation (only for room inspection) --}}
        @if($checklist->mod_type === 'room_inspection' && count($submissions) > 0)
        <div class="room-navigation">
            <div class="room-nav-title">
                <i class="fas fa-door-closed"></i>
                Jump to Room:
            </div>
            <div class="room-nav-buttons">
                @foreach($submissions as $roomNumber => $submission)
                <a href="#room-{{ $roomNumber }}" class="room-nav-btn completed">
                    <i class="fas fa-door-open"></i>
                    Room {{ $roomNumber }}
                </a>
                @endforeach
            </div>
        </div>
        @endif

        {{-- Main Content --}}
        @if(count($submissions) > 0)
            @if($checklist->mod_type === 'room_inspection')
                {{-- Display each room separately --}}
                @foreach($submissions as $roomNumber => $submission)
                <div class="room-section" id="room-{{ $roomNumber }}">
                    <div class="room-header">
                        <h2>
                            <span><i class="fas fa-door-closed"></i> Room {{ $roomNumber }}</span>
                            <span class="room-badge">
                                <i class="fas fa-check-circle"></i>
                                Completed
                            </span>
                        </h2>
                        <div class="room-meta">
                            <div class="room-meta-item">
                                <i class="fas fa-calendar-alt"></i>
                                Submitted: {{ $submission->created_at->format('h:i A') }}
                            </div>
                            <div class="room-meta-item">
                                <i class="fas fa-hashtag"></i>
                                Room Number: {{ $roomNumber }}
                            </div>
                            <div class="room-meta-item">
                                <i class="fas fa-sort-numeric-up"></i>
                                Room {{ $loop->iteration }} of {{ count($submissions) }}
                            </div>
                        </div>
                    </div>
                    
                    <div class="questions-container">
                        @if($sections->isEmpty())
                            <div class="empty-state">
                                <i class="fas fa-info-circle"></i>
                                <h3>No Sections Found</h3>
                                <p>This checklist has no sections defined.</p>
                            </div>
                        @else
                            {{-- Loop through sections --}}
                            @foreach($sections as $section)
                                @php
                                    $sectionQuestions = $section->questions ?? collect();
                                @endphp
                                
                                @if($sectionQuestions->isNotEmpty())
                                <div class="section-card">
                                    <div class="section-card-header">
                                        <i class="fas fa-folder-open"></i>
                                        {{ $section->name }}
                                    </div>
                                    
                                    {{-- Loop through questions --}}
                                    @foreach($sectionQuestions as $question)
                                    @php
                                        $answer = $allQuestionAnswers[$submission->id][$question->id] ?? null;
                                    @endphp
                                    
                                    @if(in_array($question->type, ['header', 'readOnlyText', 'pageNote']))
                                        {{-- Display special question types --}}
                                        @if($question->type === 'header')
                                        <div class="question-card" style="background: #e0f2fe; border-left: 5px solid #0ea5e9;">
                                            <div class="question-text" style="font-size: 1.2rem; color: #0369a1;">
                                                <i class="fas fa-heading"></i>
                                                {{ $question->question_text }}
                                            </div>
                                        </div>
                                        @elseif($question->type === 'readOnlyText')
                                        <div class="question-card" style="background: #f8fafc;">
                                            <div class="question-text" style="color: #64748b; font-weight: normal;">
                                                <i class="fas fa-paragraph"></i>
                                                {{ $question->question_text }}
                                            </div>
                                        </div>
                                        @elseif($question->type === 'pageNote')
                                        <div class="question-card" style="background: #f0fdf4; border-left: 5px solid #10b981;">
                                            <div class="question-text" style="color: #065f46;">
                                                <i class="fas fa-sticky-note"></i>
                                                {{ $question->question_text }}
                                            </div>
                                        </div>
                                        @endif
                                    @else
                                        {{-- Display regular questions with answers --}}
                                        <div class="question-card">
                                            <div class="question-text">
                                                {{ $question->question_text }}
                                                @if($question->is_required)
                                                <span style="color: #ef4444; margin-left: 5px;">*</span>
                                                @endif
                                            </div>
                                            
                                            {{-- Answer display --}}
                                            @if($answer && $answer->answer_yes_no)
                                            <div class="answer-display">
                                                <span class="answer-badge {{ $answer->answer_yes_no }}">
                                                    @if($answer->answer_yes_no === 'yes')
                                                        <i class="fas fa-check-circle"></i> Yes
                                                    @elseif($answer->answer_yes_no === 'no')
                                                        <i class="fas fa-times-circle"></i> No
                                                    @elseif($answer->answer_yes_no === 'good')
                                                        <i class="fas fa-thumbs-up"></i> Good
                                                    @elseif($answer->answer_yes_no === 'bad')
                                                        <i class="fas fa-thumbs-down"></i> Bad
                                                    @endif
                                                </span>
                                            </div>
                                            @elseif($answer && $question->type === 'imageUpload')
                                                <div class="answer-display">
                                                    <span class="answer-badge" style="background: #f3f4f6; color: #6b7280;">
                                                        <i class="fas fa-image"></i> Image Uploaded
                                                    </span>
                                                </div>
                                            <!-- @else
                                                <div class="answer-display">
                                                    <span class="answer-badge" style="background: #f3f4f6; color: #6b7280;">
                                                        <i class="fas fa-question-circle"></i> No Answer
                                                    </span>
                                                </div>
                                            @endif -->
                                            
                                            {{-- Comment display --}}
                                            @if($answer && $answer->comment)
                                            <div class="comment-box">
                                                <div class="comment-label">
                                                    <i class="fas fa-comment"></i> Comment:
                                                </div>
                                                <div class="comment-text">
                                                    {{ $answer->comment }}
                                                </div>
                                            </div>
                                            @endif
                                            
                                            {{-- Image display --}}
                                            @if($answer && $answer->image_path)
                                            <div class="image-display">
                                                <div class="image-display-label">
                                                    <i class="fas fa-image"></i> Photo:
                                                </div>
                                                <div class="image-container">
                                                    <img src="{{ Storage::url($answer->image_path) }}" 
                                                         alt="Question photo"
                                                         onclick="openImageModal(this.src)">
                                                </div>
                                            </div>
                                            @endif
                                            
                                            {{-- Sub questions --}}
                                            @if($question->type === 'subQuestionList' && $question->subQuestions && $question->subQuestions->count() > 0)
                                            <div class="sub-question-container">
                                                <div class="comment-label" style="margin-bottom: 15px;">
                                                    <i class="fas fa-list-ul"></i> Sub Questions:
                                                </div>
                                                
                                                @foreach($question->subQuestions as $subQuestion)
                                                @php
                                                    $subAnswer = $allSubQuestionAnswers[$submission->id][$subQuestion->id] ?? null;
                                                @endphp
                                                
                                                <div class="sub-question">
                                                    <div class="sub-question-text">
                                                        {{ $subQuestion->text }}
                                                        @if($subQuestion->is_required)
                                                        <span style="color: #ef4444; margin-left: 5px;">*</span>
                                                        @endif
                                                    </div>
                                                    
                                                    @if($subAnswer)
                                                        {{-- Sub question answer --}}
                                                        @if($subAnswer->answer_yes_no)
                                                        <div class="answer-display">
                                                            <span class="answer-badge {{ $subAnswer->answer_yes_no }}">
                                                                @if($subAnswer->answer_yes_no === 'yes')
                                                                    <i class="fas fa-check-circle"></i> Yes
                                                                @elseif($subAnswer->answer_yes_no === 'no')
                                                                    <i class="fas fa-times-circle"></i> No
                                                                @elseif($subAnswer->answer_yes_no === 'good')
                                                                    <i class="fas fa-thumbs-up"></i> Good
                                                                @elseif($subAnswer->answer_yes_no === 'bad')
                                                                    <i class="fas fa-thumbs-down"></i> Bad
                                                                @endif
                                                            </span>
                                                        </div>
                                                        @endif
                                                        
                                                        {{-- Sub question comment --}}
                                                        @if($subAnswer->comment)
                                                        <div class="comment-box" style="margin-top: 10px; background: #fff;">
                                                            <div class="comment-label" style="font-size: 0.85rem;">
                                                                <i class="fas fa-comment"></i> Comment:
                                                            </div>
                                                            <div class="comment-text" style="font-size: 0.9rem;">
                                                                {{ $subAnswer->comment }}
                                                            </div>
                                                        </div>
                                                        @endif
                                                        
                                                        {{-- Sub question image --}}
                                                        @if($subAnswer->image_path)
                                                        <div class="image-display" style="margin-top: 10px;">
                                                            <div class="image-container" style="max-width: 300px;">
                                                                <img src="{{ Storage::url($subAnswer->image_path) }}" 
                                                                     alt="Sub question photo"
                                                                     onclick="openImageModal(this.src)">
                                                            </div>
                                                        </div>
                                                        @endif
                                                    <!-- @else
                                                        <div class="comment-text" style="color: #9ca3af; font-style: italic;">
                                                            No answer provided
                                                        </div>
                                                    @endif -->
                                                </div>
                                                @endforeach
                                            </div>
                                            @endif
                                        </div>
                                    @endif
                                    @endforeach
                                </div>
                                @endif
                            @endforeach
                        @endif
                        
                        {{-- Overall comment for this room --}}
                        @if($submission->overall_comment)
                        <div class="overall-comment-section">
                            <div class="overall-comment-header">
                                <i class="fas fa-star"></i>
                                <h3>Overall Comment for Room {{ $roomNumber }}</h3>
                            </div>
                            <div class="overall-comment-text">
                                {{ $submission->overall_comment }}
                            </div>
                        </div>
                        @endif
                    </div>
                </div>
                @endforeach
                
            @else
                {{-- For General Operation (single submission) --}}
                @php
                    $submission = reset($submissions);
                @endphp
                
                <div class="room-section">
                    <div class="room-header" style="background: linear-gradient(to right, #10b981, #34d399);">
                        <h2>
                            <i class="fas fa-clipboard-list"></i>
                            General Operation Review
                        </h2>
                        <div class="room-meta">
                            <div class="room-meta-item">
                                <i class="fas fa-calendar-alt"></i>
                                Submitted: {{ $submission->created_at->format('h:i A') }}
                            </div>
                            <div class="room-meta-item">
                                <i class="fas fa-user"></i>
                                Inspector: {{ auth()->user()->name }}
                            </div>
                            <div class="room-meta-item">
                                <i class="fas fa-hotel"></i>
                                Hotel: {{ $checklist->hotel }}
                            </div>
                        </div>
                    </div>
                    
                    <div class="questions-container">
                        @if($sections->isEmpty())
                            <div class="empty-state">
                                <i class="fas fa-info-circle"></i>
                                <h3>No Sections Found</h3>
                                <p>This checklist has no sections defined.</p>
                            </div>
                        @else
                            {{-- Loop through sections --}}
                            @foreach($sections as $section)
                                @php
                                    $sectionQuestions = $section->questions ?? collect();
                                @endphp
                                
                                @if($sectionQuestions->isNotEmpty())
                                <div class="section-card">
                                    <div class="section-card-header">
                                        <i class="fas fa-folder-open"></i>
                                        {{ $section->name }}
                                    </div>
                                    
                                    {{-- Loop through questions --}}
                                    @foreach($sectionQuestions as $question)
                                    @php
                                        $answer = $allQuestionAnswers[$submission->id][$question->id] ?? null;
                                    @endphp
                                    
                                    @if(in_array($question->type, ['header', 'readOnlyText', 'pageNote']))
                                        {{-- Display special question types --}}
                                        @if($question->type === 'header')
                                        <div class="question-card" style="background: #e0f2fe; border-left: 5px solid #0ea5e9;">
                                            <div class="question-text" style="font-size: 1.2rem; color: #0369a1;">
                                                <i class="fas fa-heading"></i>
                                                {{ $question->question_text }}
                                            </div>
                                        </div>
                                        @elseif($question->type === 'readOnlyText')
                                        <div class="question-card" style="background: #f8fafc;">
                                            <div class="question-text" style="color: #64748b; font-weight: normal;">
                                                <i class="fas fa-paragraph"></i>
                                                {{ $question->question_text }}
                                            </div>
                                        </div>
                                        @elseif($question->type === 'pageNote')
                                        <div class="question-card" style="background: #f0fdf4; border-left: 5px solid #10b981;">
                                            <div class="question-text" style="color: #065f46;">
                                                <i class="fas fa-sticky-note"></i>
                                                {{ $question->question_text }}
                                            </div>
                                        </div>
                                        @endif
                                    @else
                                        {{-- Display regular questions with answers --}}
                                        <div class="question-card">
                                            <div class="question-text">
                                                {{ $question->question_text }}
                                                @if($question->is_required)
                                                <span style="color: #ef4444; margin-left: 5px;">*</span>
                                                @endif
                                            </div>
                                            
                                            {{-- Answer display --}}
                                            @if($answer && $answer->answer_yes_no)
                                            <div class="answer-display">
                                                <span class="answer-badge {{ $answer->answer_yes_no }}">
                                                    @if($answer->answer_yes_no === 'yes')
                                                        <i class="fas fa-check-circle"></i> Yes
                                                    @elseif($answer->answer_yes_no === 'no')
                                                        <i class="fas fa-times-circle"></i> No
                                                    @elseif($answer->answer_yes_no === 'good')
                                                        <i class="fas fa-thumbs-up"></i> Good
                                                    @elseif($answer->answer_yes_no === 'bad')
                                                        <i class="fas fa-thumbs-down"></i> Bad
                                                    @endif
                                                </span>
                                            </div>
                                            @elseif($answer && $question->type === 'imageUpload')
                                                <div class="answer-display">
                                                    <span class="answer-badge" style="background: #f3f4f6; color: #6b7280;">
                                                        <i class="fas fa-image"></i> Image Uploaded
                                                    </span>
                                                </div>
                                            <!-- @else
                                                <div class="answer-display">
                                                    <span class="answer-badge" style="background: #f3f4f6; color: #6b7280;">
                                                        <i class="fas fa-question-circle"></i> No Answer
                                                    </span>
                                                </div>
                                            @endif -->
                                            
                                            {{-- Comment display --}}
                                            @if($answer && $answer->comment)
                                            <div class="comment-box">
                                                <div class="comment-label">
                                                    <i class="fas fa-comment"></i> Comment:
                                                </div>
                                                <div class="comment-text">
                                                    {{ $answer->comment }}
                                                </div>
                                            </div>
                                            @endif
                                            
                                            {{-- Image display --}}
                                            @if($answer && $answer->image_path)
                                            <div class="image-display">
                                                <div class="image-display-label">
                                                    <i class="fas fa-image"></i> Photo:
                                                </div>
                                                <div class="image-container">
                                                    <img src="{{ Storage::url($answer->image_path) }}" 
                                                         alt="Question photo"
                                                         onclick="openImageModal(this.src)">
                                                </div>
                                            </div>
                                            @endif
                                            
                                            {{-- Sub questions --}}
                                            @if($question->type === 'subQuestionList' && $question->subQuestions && $question->subQuestions->count() > 0)
                                            <div class="sub-question-container">
                                                <div class="comment-label" style="margin-bottom: 15px;">
                                                    <i class="fas fa-list-ul"></i> Sub Questions:
                                                </div>
                                                
                                                @foreach($question->subQuestions as $subQuestion)
                                                @php
                                                    $subAnswer = $allSubQuestionAnswers[$submission->id][$subQuestion->id] ?? null;
                                                @endphp
                                                
                                                <div class="sub-question">
                                                    <div class="sub-question-text">
                                                        {{ $subQuestion->text }}
                                                        @if($subQuestion->is_required)
                                                        <span style="color: #ef4444; margin-left: 5px;">*</span>
                                                        @endif
                                                    </div>
                                                    
                                                    @if($subAnswer)
                                                        {{-- Sub question answer --}}
                                                        @if($subAnswer->answer_yes_no)
                                                        <div class="answer-display">
                                                            <span class="answer-badge {{ $subAnswer->answer_yes_no }}">
                                                                @if($subAnswer->answer_yes_no === 'yes')
                                                                    <i class="fas fa-check-circle"></i> Yes
                                                                @elseif($subAnswer->answer_yes_no === 'no')
                                                                    <i class="fas fa-times-circle"></i> No
                                                                @elseif($subAnswer->answer_yes_no === 'good')
                                                                    <i class="fas fa-thumbs-up"></i> Good
                                                                @elseif($subAnswer->answer_yes_no === 'bad')
                                                                    <i class="fas fa-thumbs-down"></i> Bad
                                                                @endif
                                                            </span>
                                                        </div>
                                                        @endif
                                                        
                                                        {{-- Sub question comment --}}
                                                        @if($subAnswer->comment)
                                                        <div class="comment-box" style="margin-top: 10px; background: #fff;">
                                                            <div class="comment-label" style="font-size: 0.85rem;">
                                                                <i class="fas fa-comment"></i> Comment:
                                                            </div>
                                                            <div class="comment-text" style="font-size: 0.9rem;">
                                                                {{ $subAnswer->comment }}
                                                            </div>
                                                        </div>
                                                        @endif
                                                        
                                                        {{-- Sub question image --}}
                                                        @if($subAnswer->image_path)
                                                        <div class="image-display" style="margin-top: 10px;">
                                                            <div class="image-container" style="max-width: 300px;">
                                                                <img src="{{ Storage::url($subAnswer->image_path) }}" 
                                                                     alt="Sub question photo"
                                                                     onclick="openImageModal(this.src)">
                                                            </div>
                                                        </div>
                                                        @endif
                                                    <!-- @else
                                                        <div class="comment-text" style="color: #9ca3af; font-style: italic;">
                                                            No answer provided
                                                        </div>
                                                    @endif -->
                                                </div>
                                                @endforeach
                                            </div>
                                            @endif
                                        </div>
                                    @endif
                                    @endforeach
                                </div>
                                @endif
                            @endforeach
                        @endif
                    </div>
                </div>
            @endif
            
            {{-- Print and Navigation Buttons --}}
<div class="review-navigation">
    <a href="{{ route('user.dashboard') }}" class="nav-btn btn-back">
        <i class="fas fa-arrow-left"></i>
        Back to Dashboard
    </a>
    
    <a href="{{ route('user.checklist.report', $checklist->id) }}" class="nav-btn btn-print" target="_blank">
        <i class="fas fa-file-alt"></i>
        View Full Report
    </a>
</div>
            
        @else
            {{-- Empty state --}}
            <div class="empty-state">
                <i class="fas fa-clipboard-question"></i>
                <h3>No Submissions Found</h3>
                <p>You haven't completed any inspections for today yet. Start your first inspection to see the review here.</p>
                <a href="{{ route('user.checklist.fill', ['checklistId' => $checklist->id, 'step' => 0]) }}" 
                   class="nav-btn btn-edit" style="display: inline-flex;">
                    <i class="fas fa-play-circle"></i>
                    Start Inspection
                </a>
            </div>
        @endif
    </div>
</div>
@endif
@endsection

{{-- ================= MOBILE SECTION ================= --}}
@section('mobile-content')
@php
    // Default values to prevent undefined variable errors
    $checklist = $checklist ?? null;
    $submissions = $submissions ?? [];
    $sections = $sections ?? collect();
    $allQuestionAnswers = $allQuestionAnswers ?? [];
    $allSubQuestionAnswers = $allSubQuestionAnswers ?? [];
@endphp

@if(!$checklist)
    <div class="mobile-section">
        <div style="padding: 20px;">
            <div style="padding: 20px; background: #fee2e2; color: #991b1b; border-radius: 10px;">
                <h3 style="margin-top: 0;"><i class="fas fa-exclamation-triangle"></i> Checklist Not Found</h3>
                <p>The checklist you're trying to review does not exist or has been deleted.</p>
                <a href="{{ route('user.dashboard') }}" style="display: inline-block; padding: 10px 20px; background: #f3f4f6; color: #374151; border-radius: 8px; text-decoration: none; margin-top: 15px;">
                    <i class="fas fa-arrow-left"></i> Back to Dashboard
                </a>
            </div>
        </div>
    </div>
@else
<div class="mobile-section">
    <div class="review-container" style="padding: 0;">
        {{-- Header Mobile --}}
        <div class="review-header mobile-review-header">
            <h1 style="font-size: 1.6rem;">
                <i class="fas fa-clipboard-check"></i>
                {{ $checklist->hotel }} - {{ $checklist->title }} Review
            </h1>
            <div class="subtitle" style="font-size: 0.95rem;">
                {{ now()->format('F j, Y') }}
            </div>
            
            <div class="mobile-stats-grid">
                <div class="mobile-stat-box">
                    <div class="stat-label">Total {{ $checklist->mod_type === 'room_inspection' ? 'Rooms' : 'Checklists' }}</div>
                    <div class="stat-value">
                        @if($checklist->mod_type === 'room_inspection')
                            {{ count($submissions) }}
                        @else
                            {{ count($submissions) > 0 ? 1 : 0 }}
                        @endif
                    </div>
                </div>
                <div class="mobile-stat-box">
                    <div class="stat-label">Hotel</div>
                    <div class="stat-value">{{ $checklist->hotel }}</div>
                </div>
                <div class="mobile-stat-box">
                    <div class="stat-label">Type</div>
                    <div class="stat-value">
                        @if($checklist->mod_type === 'room_inspection')
                            Room
                        @else
                            General
                        @endif
                    </div>
                </div>
                <div class="mobile-stat-box">
                    <div class="stat-label">Date</div>
                    <div class="stat-value">
                        {{ now()->format('M d') }}
                    </div>
                </div>
            </div>
        </div>

        {{-- Room Navigation Mobile --}}
        @if($checklist->mod_type === 'room_inspection' && count($submissions) > 0)
        <div class="mobile-room-navigation">
            <div class="room-nav-title" style="margin-bottom: 10px;">
                <i class="fas fa-door-closed"></i> Jump to Room:
            </div>
            <div class="mobile-room-nav-buttons">
                @foreach($submissions as $roomNumber => $submission)
                <a href="#mobile-room-{{ $roomNumber }}" class="mobile-room-nav-btn completed">
                    Room {{ $roomNumber }}
                </a>
                @endforeach
            </div>
        </div>
        @endif

        {{-- Main Content Mobile --}}
        @if(count($submissions) > 0)
            @if($checklist->mod_type === 'room_inspection')
                {{-- Display each room separately --}}
                @foreach($submissions as $roomNumber => $submission)
                <div class="room-section" id="mobile-room-{{ $roomNumber }}" style="margin-bottom: 25px;">
                    <div class="room-header mobile-room-header">
                        <h2>
                            <span><i class="fas fa-door-closed"></i> Room {{ $roomNumber }}</span>
                            <span class="room-badge">
                                <i class="fas fa-check-circle"></i>
                                Completed
                            </span>
                        </h2>
                        <div class="mobile-room-meta">
                            <div class="mobile-room-meta-item">
                                <i class="fas fa-calendar-alt"></i>
                                {{ $submission->created_at->format('h:i A') }}
                            </div>
                            <div class="mobile-room-meta-item">
                                <i class="fas fa-hashtag"></i>
                                Room {{ $roomNumber }}
                            </div>
                            <div class="mobile-room-meta-item">
                                <i class="fas fa-sort-numeric-up"></i>
                                {{ $loop->iteration }}/{{ count($submissions) }}
                            </div>
                        </div>
                    </div>
                    
                    <div class="questions-container mobile-questions-container">
                        @if($sections->isEmpty())
                            <div class="empty-state" style="padding: 40px 20px;">
                                <i class="fas fa-info-circle"></i>
                                <h3>No Sections</h3>
                            </div>
                        @else
                            @foreach($sections as $section)
                                @php
                                    $sectionQuestions = $section->questions ?? collect();
                                @endphp
                                
                                @if($sectionQuestions->isNotEmpty())
                                <div class="section-card mobile-section-card">
                                    <div class="section-card-header mobile-section-card-header">
                                        <i class="fas fa-folder-open"></i>
                                        {{ $section->name }}
                                    </div>
                                    
                                    @foreach($sectionQuestions as $question)
                                    @php
                                        $answer = $allQuestionAnswers[$submission->id][$question->id] ?? null;
                                    @endphp
                                    
                                    @if(in_array($question->type, ['header', 'readOnlyText', 'pageNote']))
                                        @if($question->type === 'header')
                                        <div class="question-card mobile-question-card" style="background: #e0f2fe; border-left: 5px solid #0ea5e9;">
                                            <div class="question-text mobile-question-text" style="font-size: 1.1rem; color: #0369a1;">
                                                <i class="fas fa-heading"></i>
                                                {{ $question->question_text }}
                                            </div>
                                        </div>
                                        @elseif($question->type === 'readOnlyText')
                                        <div class="question-card mobile-question-card" style="background: #f8fafc;">
                                            <div class="question-text mobile-question-text" style="color: #64748b; font-weight: normal;">
                                                <i class="fas fa-paragraph"></i>
                                                {{ $question->question_text }}
                                            </div>
                                        </div>
                                        @elseif($question->type === 'pageNote')
                                        <div class="question-card mobile-question-card" style="background: #f0fdf4; border-left: 5px solid #10b981;">
                                            <div class="question-text mobile-question-text" style="color: #065f46;">
                                                <i class="fas fa-sticky-note"></i>
                                                {{ $question->question_text }}
                                            </div>
                                        </div>
                                        @endif
                                    @else
                                        <div class="question-card mobile-question-card">
                                            <div class="question-text mobile-question-text">
                                                {{ $question->question_text }}
                                                @if($question->is_required)
                                                <span style="color: #ef4444; margin-left: 5px;">*</span>
                                                @endif
                                            </div>
                                            
                                            @if($answer && $answer->answer_yes_no)
                                            <div class="answer-display">
                                                <span class="answer-badge mobile-answer-badge {{ $answer->answer_yes_no }}">
                                                    @if($answer->answer_yes_no === 'yes')
                                                        <i class="fas fa-check-circle"></i> Yes
                                                    @elseif($answer->answer_yes_no === 'no')
                                                        <i class="fas fa-times-circle"></i> No
                                                    @elseif($answer->answer_yes_no === 'good')
                                                        <i class="fas fa-thumbs-up"></i> Good
                                                    @elseif($answer->answer_yes_no === 'bad')
                                                        <i class="fas fa-thumbs-down"></i> Bad
                                                    @endif
                                                </span>
                                            </div>
                                            @elseif($answer && $question->type === 'imageUpload')
                                                <div class="answer-display">
                                                    <span class="answer-badge mobile-answer-badge" style="background: #f3f4f6; color: #6b7280;">
                                                        <i class="fas fa-image"></i> Image
                                                    </span>
                                                </div>
                                            <!-- @else
                                                <div class="answer-display">
                                                    <span class="answer-badge mobile-answer-badge" style="background: #f3f4f6; color: #6b7280;">
                                                        <i class="fas fa-question-circle"></i> No Answer
                                                    </span>
                                                </div>
                                            @endif -->
                                            
                                            @if($answer && $answer->comment)
                                            <div class="comment-box mobile-comment-box">
                                                <div class="comment-label" style="font-size: 0.85rem;">
                                                    <i class="fas fa-comment"></i> Comment:
                                                </div>
                                                <div class="comment-text" style="font-size: 0.85rem;">
                                                    {{ $answer->comment }}
                                                </div>
                                            </div>
                                            @endif
                                            
                                            @if($answer && $answer->image_path)
                                            <div class="image-display">
                                                <div class="image-display-label" style="font-size: 0.85rem;">
                                                    <i class="fas fa-image"></i> Photo:
                                                </div>
                                                <div class="image-container mobile-image-container">
                                                    <img src="{{ Storage::url($answer->image_path) }}" 
                                                         alt="Question photo"
                                                         onclick="openImageModal(this.src)">
                                                </div>
                                            </div>
                                            @endif
                                            
                                            @if($question->type === 'subQuestionList' && $question->subQuestions && $question->subQuestions->count() > 0)
                                            <div class="sub-question-container" style="padding: 12px;">
                                                <div class="comment-label" style="margin-bottom: 12px; font-size: 0.85rem;">
                                                    <i class="fas fa-list-ul"></i> Sub Questions:
                                                </div>
                                                
                                                @foreach($question->subQuestions as $subQuestion)
                                                @php
                                                    $subAnswer = $allSubQuestionAnswers[$submission->id][$subQuestion->id] ?? null;
                                                @endphp
                                                
                                                <div class="sub-question" style="padding: 12px;">
                                                    <div class="sub-question-text" style="font-size: 0.9rem;">
                                                        {{ $subQuestion->text }}
                                                        @if($subQuestion->is_required)
                                                        <span style="color: #ef4444;">*</span>
                                                        @endif
                                                    </div>
                                                    
                                                    @if($subAnswer)
                                                        @if($subAnswer->answer_yes_no)
                                                        <div class="answer-display">
                                                            <span class="answer-badge mobile-answer-badge {{ $subAnswer->answer_yes_no }}" style="font-size: 0.75rem;">
                                                                @if($subAnswer->answer_yes_no === 'yes') Yes
                                                                @elseif($subAnswer->answer_yes_no === 'no') No
                                                                @elseif($subAnswer->answer_yes_no === 'good') Good
                                                                @elseif($subAnswer->answer_yes_no === 'bad') Bad
                                                                @endif
                                                            </span>
                                                        </div>
                                                        @endif
                                                        
                                                        @if($subAnswer->comment)
                                                        <div class="comment-box" style="margin-top: 8px; padding: 10px;">
                                                            <div class="comment-text" style="font-size: 0.8rem;">
                                                                {{ $subAnswer->comment }}
                                                            </div>
                                                        </div>
                                                        @endif
                                                        
                                                        @if($subAnswer->image_path)
                                                        <div class="image-display" style="margin-top: 8px;">
                                                            <div class="image-container" style="max-width: 200px;">
                                                                <img src="{{ Storage::url($subAnswer->image_path) }}" 
                                                                     alt="Sub question photo"
                                                                     onclick="openImageModal(this.src)">
                                                            </div>
                                                        </div>
                                                        @endif
                                                    <!-- @else
                                                        <div class="comment-text" style="color: #9ca3af; font-style: italic; font-size: 0.8rem;">
                                                            No answer provided
                                                        </div>
                                                    @endif -->
                                                </div>
                                                @endforeach
                                            </div>
                                            @endif
                                        </div>
                                    @endif
                                    @endforeach
                                </div>
                                @endif
                            @endforeach
                        @endif
                        
                        @if($submission->overall_comment)
                        <div class="overall-comment-section mobile-overall-comment-section">
                            <div class="overall-comment-header mobile-overall-comment-header">
                                <i class="fas fa-star" style="font-size: 1.2rem;"></i>
                                <h3 style="font-size: 1.1rem;">Overall Comment</h3>
                            </div>
                            <div class="overall-comment-text mobile-overall-comment-text">
                                {{ $submission->overall_comment }}
                            </div>
                        </div>
                        @endif
                    </div>
                </div>
                @endforeach
                
            @else
                {{-- General Operation Mobile --}}
                @php
                    $submission = reset($submissions);
                @endphp
                
                <div class="room-section">
                    <div class="room-header mobile-room-header" style="background: linear-gradient(to right, #10b981, #34d399);">
                        <h2>
                            <i class="fas fa-clipboard-list"></i>
                            General Operation
                        </h2>
                        <div class="mobile-room-meta">
                            <div class="mobile-room-meta-item">
                                <i class="fas fa-calendar-alt"></i>
                                {{ $submission->created_at->format('h:i A') }}
                            </div>
                            <div class="mobile-room-meta-item">
                                <i class="fas fa-user"></i>
                                {{ auth()->user()->name }}
                            </div>
                        </div>
                    </div>
                    
                    <div class="questions-container mobile-questions-container">
                        @if($sections->isEmpty())
                            <div class="empty-state" style="padding: 40px 20px;">
                                <i class="fas fa-info-circle"></i>
                                <h3>No Sections</h3>
                            </div>
                        @else
                            @foreach($sections as $section)
                                @php
                                    $sectionQuestions = $section->questions ?? collect();
                                @endphp
                                
                                @if($sectionQuestions->isNotEmpty())
                                <div class="section-card mobile-section-card">
                                    <div class="section-card-header mobile-section-card-header">
                                        <i class="fas fa-folder-open"></i>
                                        {{ $section->name }}
                                    </div>
                                    
                                    @foreach($sectionQuestions as $question)
                                    @php
                                        $answer = $allQuestionAnswers[$submission->id][$question->id] ?? null;
                                    @endphp
                                    
                                    {{-- Sama seperti di atas, gunakan kode question yang sama --}}
                                    @if(in_array($question->type, ['header', 'readOnlyText', 'pageNote']))
                                        @if($question->type === 'header')
                                        <div class="question-card mobile-question-card" style="background: #e0f2fe;">
                                            <div class="question-text mobile-question-text" style="color: #0369a1;">
                                                <i class="fas fa-heading"></i> {{ $question->question_text }}
                                            </div>
                                        </div>
                                        @elseif($question->type === 'readOnlyText')
                                        <div class="question-card mobile-question-card" style="background: #f8fafc;">
                                            <div class="question-text mobile-question-text" style="color: #64748b; font-weight: normal;">
                                                {{ $question->question_text }}
                                            </div>
                                        </div>
                                        @elseif($question->type === 'pageNote')
                                        <div class="question-card mobile-question-card" style="background: #f0fdf4;">
                                            <div class="question-text mobile-question-text" style="color: #065f46;">
                                                {{ $question->question_text }}
                                            </div>
                                        </div>
                                        @endif
                                    @else
                                        <div class="question-card mobile-question-card">
                                            <div class="question-text mobile-question-text">
                                                {{ $question->question_text }}
                                            </div>
                                            
                                            @if($answer && $answer->answer_yes_no)
                                            <div class="answer-display">
                                                <span class="answer-badge mobile-answer-badge {{ $answer->answer_yes_no }}">
                                                    {{ ucfirst($answer->answer_yes_no) }}
                                                </span>
                                            </div>
                                            <!-- @else
                                                <div class="answer-display">
                                                    <span class="answer-badge mobile-answer-badge" style="background: #f3f4f6; color: #6b7280;">
                                                        No Answer
                                                    </span>
                                                </div>
                                            @endif -->
                                            
                                            @if($answer && $answer->comment)
                                            <div class="comment-box mobile-comment-box">
                                                <div class="comment-text" style="font-size: 0.85rem;">
                                                    {{ $answer->comment }}
                                                </div>
                                            </div>
                                            @endif
                                            
                                            @if($answer && $answer->image_path)
                                            <div class="image-display">
                                                <div class="image-container mobile-image-container">
                                                    <img src="{{ Storage::url($answer->image_path) }}" 
                                                         alt="Photo"
                                                         onclick="openImageModal(this.src)">
                                                </div>
                                            </div>
                                            @endif
                                            
                                            @if($question->type === 'subQuestionList' && $question->subQuestions && $question->subQuestions->count() > 0)
                                            <div class="sub-question-container" style="padding: 12px;">
                                                @foreach($question->subQuestions as $subQuestion)
                                                @php
                                                    $subAnswer = $allSubQuestionAnswers[$submission->id][$subQuestion->id] ?? null;
                                                @endphp
                                                <div class="sub-question" style="padding: 12px;">
                                                    <div class="sub-question-text" style="font-size: 0.9rem;">
                                                        {{ $subQuestion->text }}
                                                    </div>
                                                    @if($subAnswer)
                                                        @if($subAnswer->answer_yes_no)
                                                        <span class="answer-badge mobile-answer-badge {{ $subAnswer->answer_yes_no }}" style="font-size: 0.75rem;">
                                                            {{ ucfirst($subAnswer->answer_yes_no) }}
                                                        </span>
                                                        @endif
                                                        @if($subAnswer->comment)
                                                        <div style="margin-top: 6px; font-size: 0.8rem;">{{ $subAnswer->comment }}</div>
                                                        @endif
                                                        @if($subAnswer->image_path)
                                                        <div style="margin-top: 8px;">
                                                            <img src="{{ Storage::url($subAnswer->image_path) }}" style="max-width: 150px; border-radius: 6px;">
                                                        </div>
                                                        @endif
                                                    <!-- @else
                                                        <div style="color: #9ca3af; font-style: italic; font-size: 0.8rem;">No answer</div>
                                                    @endif -->
                                                </div>
                                                @endforeach
                                            </div>
                                            @endif
                                        </div>
                                    @endif
                                    @endforeach
                                </div>
                                @endif
                            @endforeach
                        @endif
                    </div>
                </div>
            @endif
            
           {{-- Mobile Navigation Buttons --}}
<div class="review-navigation mobile-review-navigation">
    <a href="{{ route('user.dashboard') }}" class="nav-btn btn-back mobile-nav-btn">
        <i class="fas fa-arrow-left"></i>
        Back to Dashboard
    </a>
    
    <a href="{{ route('user.checklist.report', $checklist->id) }}" class="nav-btn btn-print mobile-nav-btn" target="_blank">
        <i class="fas fa-file-alt"></i>
        View Report
    </a>
</div>
            
        @else
            <div class="empty-state" style="padding: 40px 20px;">
                <i class="fas fa-clipboard-question"></i>
                <h3>No Submissions Found</h3>
                <p>You haven't completed any inspections for today yet.</p>
                <a href="{{ route('user.checklist.fill', ['checklistId' => $checklist->id, 'step' => 0]) }}" 
                   style="display: inline-block; padding: 12px 24px; background: #3b82f6; color: white; border-radius: 8px; text-decoration: none; margin-top: 15px;">
                    <i class="fas fa-play-circle"></i>
                    Start Inspection
                </a>
            </div>
        @endif
    </div>
</div>
@endif
@endsection

{{-- ================= IMAGE MODAL & SCRIPTS ================= --}}
@push('scripts')
<script>
// Image modal functionality
function openImageModal(src) {
    // Create modal if not exists
    let modal = document.getElementById('imageModal');
    if (!modal) {
        modal = document.createElement('div');
        modal.id = 'imageModal';
        modal.className = 'image-modal';
        modal.innerHTML = `
            <button class="image-modal-close" onclick="closeImageModal()">&times;</button>
            <div class="image-modal-content">
                <img id="modalImage" src="" alt="Full size image">
            </div>
        `;
        document.body.appendChild(modal);
    }
    
    document.getElementById('modalImage').src = src;
    modal.style.display = 'flex';
    document.body.style.overflow = 'hidden';
}

function closeImageModal() {
    const modal = document.getElementById('imageModal');
    if (modal) {
        modal.style.display = 'none';
        document.body.style.overflow = 'auto';
    }
}

// Close modal on ESC key
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        closeImageModal();
    }
});

// Smooth scroll for room navigation
document.addEventListener('DOMContentLoaded', function() {
    // Desktop
    document.querySelectorAll('.room-nav-btn').forEach(btn => {
        btn.addEventListener('click', function(e) {
            e.preventDefault();
            const targetId = this.getAttribute('href');
            const targetElement = document.querySelector(targetId);
            if (targetElement) {
                targetElement.scrollIntoView({
                    behavior: 'smooth',
                    block: 'start'
                });
                
                document.querySelectorAll('.room-nav-btn').forEach(b => {
                    b.classList.remove('active');
                });
                this.classList.add('active');
            }
        });
    });
    
    // Mobile
    document.querySelectorAll('.mobile-room-nav-btn').forEach(btn => {
        btn.addEventListener('click', function(e) {
            e.preventDefault();
            const targetId = this.getAttribute('href');
            const targetElement = document.querySelector(targetId);
            if (targetElement) {
                targetElement.scrollIntoView({
                    behavior: 'smooth',
                    block: 'start'
                });
            }
        });
    });
});
</script>
@endpush