@php
$hideMobileHeader = true;
@endphp

@extends('layouts.UserLayout')

@section('content')
<!-- DESKTOP - TAMPILAN MENARIK -->
<div class="d-none d-lg-block">
    <!-- Hero Header dengan Animasi -->
    <div class="text-center mb-5 animated-header">
        <div class="calendar-icon mb-3">
            <i class="fas fa-calendar-alt fa-4x" style="color: var(--accent-blue);
                animation: calendarPulse 2s infinite alternate;"></i>
        </div>
        <h2 class="fw-bold mb-2 gradient-text">
            <span class="text-primary">Jadwal MOD</span>
            <span class="text-accent">{{ $currentMonth }} {{ $currentYear }}</span>
        </h2>
        <p class="text-muted fs-5">
            <i class="fas fa-calendar-check me-2" style="color: var(--success);"></i>
            Jadwal aktif untuk bulan ini
        </p>

        <!-- Status Indicator -->
        <div class="status-indicator d-inline-flex align-items-center gap-3 px-4 py-2 rounded-pill
                    bg-accent-blue-lightest border border-accent-blue" style="max-width: 300px;">
            <div class="status-dot" style="background-color: {{ $jadwal ? 'var(--success)' : 'var(--gray-400)' }};"></div>
            <span class="small fw-medium">
                {{ $jadwal ? 'Jadwal tersedia' : 'Menunggu jadwal' }}
            </span>
            <span class="badge bg-{{ $jadwal ? 'success' : 'secondary' }} rounded-pill">
                {{ $jadwal ? '✓' : '○' }}
            </span>
        </div>
    </div>

    @if($jadwal && $jadwal->image_path)
        @php
            // Cek apakah gambar benar-benar ada di storage
            $imageFullPath = storage_path('app/public/' . $jadwal->image_path);
            $imageExists = file_exists($imageFullPath);
            $imageUrl = $imageExists ? asset('storage/' . $jadwal->image_path) : null;

            // Info tambahan
            $fileSize = $imageExists ? round(filesize($imageFullPath) / 1024, 1) . ' KB' : 'N/A';
            $lastModified = $imageExists ? date('d M Y', filemtime($imageFullPath)) : 'N/A';
        @endphp

        @if($imageExists && $imageUrl)
            <!-- Container Gambar dengan Efek Modern -->
            <div class="jadwal-container">
                <!-- Header Card -->
                <div class="card border-0 shadow-lg mb-4 animate-slide-up">
                    <div class="card-body p-4">
                        <div class="row align-items-center">
                            <div class="col-md-8">
                                <div class="d-flex align-items-center gap-3">
                                    <div class="icon-wrapper bg-primary rounded-circle p-3">
                                        <i class="fas fa-calendar-day fa-lg text-white"></i>
                                    </div>
                                    <div>
                                        <h4 class="fw-bold mb-1 text-gradient">{{ $jadwal->title }}</h4>
                                        <div class="d-flex gap-3">
                                            <span class="badge bg-primary rounded-pill">
                                                <i class="fas fa-calendar me-1"></i> {{ $currentMonth }}
                                            </span>
                                            <span class="badge bg-accent-blue rounded-pill">
                                                <i class="fas fa-calendar-alt me-1"></i> {{ $currentYear }}
                                            </span>
                                            <span class="badge bg-success rounded-pill">
                                                <i class="fas fa-check-circle me-1"></i> Aktif
                                            </span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-4 text-end">
                                <div class="file-info">
                                    <p class="small text-muted mb-1">
                                        <i class="fas fa-file-image me-1"></i> PNG • {{ $fileSize }}
                                    </p>
                                    <p class="small text-muted mb-0">
                                        <i class="fas fa-clock me-1"></i> Update: {{ $lastModified }}
                                    </p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Gambar dengan Frame Modern -->
                <div class="jadwal-image-wrapper animate-fade-in">
                    <div class="image-frame shadow-xl rounded-4 overflow-hidden position-relative" id="imageContainer">
                        <div class="image-wrapper" id="imageWrapper">
                            <img src="{{ $imageUrl }}"
                                 alt="Jadwal {{ $currentMonth }} {{ $currentYear }}"
                                 class="img-fluid w-100 h-auto"
                                 id="jadwalImage"
                                 data-original-src="{{ $imageUrl }}">
                        </div>

                        <!-- Overlay Controls -->
                        <div class="image-overlay" id="imageOverlay">
                            <div class="zoom-controls">
                                <button class="btn btn-light btn-lg rounded-circle shadow-lg"
                                        onclick="zoomIn()"
                                        title="Zoom In">
                                    <i class="fas fa-search-plus"></i>
                                </button>
                                <button class="btn btn-light btn-lg rounded-circle shadow-lg mx-3"
                                        onclick="zoomOut()"
                                        title="Zoom Out">
                                    <i class="fas fa-search-minus"></i>
                                </button>
                                <button class="btn btn-light btn-lg rounded-circle shadow-lg"
                                        onclick="resetZoom()"
                                        title="Reset Zoom">
                                    <i class="fas fa-undo"></i>
                                </button>
                                <button class="btn btn-light btn-lg rounded-circle shadow-lg"
                                        onclick="toggleFullscreen()"
                                        title="Fullscreen">
                                    <i class="fas fa-expand"></i>
                                </button>
                            </div>

                            <!-- Zoom Indicator -->
                            <div class="zoom-indicator" id="zoomIndicator">
                                100%
                            </div>
                        </div>

                        <!-- Loading Animation -->
                        <div class="image-loading" id="imageLoading">
                            <div class="spinner-grow text-primary" role="status"></div>
                        </div>

                        <!-- Boundary Indicator -->
                        <div class="boundary-indicator" id="boundaryIndicator">
                            <div class="boundary-line top"></div>
                            <div class="boundary-line bottom"></div>
                            <div class="boundary-line left"></div>
                            <div class="boundary-line right"></div>
                        </div>
                    </div>

                    <!-- Image Info -->
                    <div class="image-info mt-4 text-center">
                        <div class="info-cards row g-3 justify-content-center">
                            <div class="col-auto">
                                <div class="info-card small px-3 py-2 rounded-pill bg-primary-light">
                                    <i class="fas fa-expand-alt me-1"></i>
                                    <span>Scroll untuk zoom</span>
                                </div>
                            </div>
                            <div class="col-auto">
                                <div class="info-card small px-3 py-2 rounded-pill bg-success-light">
                                    <i class="fas fa-mouse-pointer me-1"></i>
                                    <span>Drag untuk geser gambar</span>
                                </div>
                            </div>
                            <div class="col-auto">
                                <div class="info-card small px-3 py-2 rounded-pill bg-warning-light">
                                    <i class="fas fa-sync-alt me-1"></i>
                                    <span>Auto refresh 5 menit</span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Info Panel -->
                <div class="info-panel mt-5 animate-slide-up" style="animation-delay: 0.2s;">
                    <div class="card border-0 shadow-sm">
                        <div class="card-body p-4">
                            <div class="row">
                                <div class="col-md-6">
                                    <h6 class="fw-bold mb-3">
                                        <i class="fas fa-info-circle me-2 text-primary"></i>
                                        Panduan Penggunaan
                                    </h6>
                                    <ul class="list-unstyled mb-0">
                                        <li class="mb-2">
                                            <i class="fas fa-check-circle text-success me-2"></i>
                                            <span>Gunakan scroll mouse untuk zoom in/out</span>
                                        </li>
                                        <li class="mb-2">
                                            <i class="fas fa-check-circle text-success me-2"></i>
                                            <span>Klik dan drag untuk menggeser gambar</span>
                                        </li>
                                        <li class="mb-2">
                                            <i class="fas fa-check-circle text-success me-2"></i>
                                            <span>Zoom akan mengikuti posisi kursor</span>
                                        </li>
                                        <li>
                                            <i class="fas fa-check-circle text-success me-2"></i>
                                            <span>Gambar tidak akan keluar dari batas</span>
                                        </li>
                                    </ul>
                                </div>
                                <div class="col-md-6">
                                    <h6 class="fw-bold mb-3">
                                        <i class="fas fa-clock me-2 text-primary"></i>
                                        Timeline
                                    </h6>
                                    <div class="timeline">
                                        <div class="timeline-item">
                                            <div class="timeline-marker bg-primary"></div>
                                            <div class="timeline-content">
                                                <small class="text-muted">Dibuat</small>
                                                <p class="mb-0 fw-medium">{{ $jadwal->created_at->format('d M Y') }}</p>
                                            </div>
                                        </div>
                                        <div class="timeline-item">
                                            <div class="timeline-marker bg-success"></div>
                                            <div class="timeline-content">
                                                <small class="text-muted">Terakhir dilihat</small>
                                                <p class="mb-0 fw-medium">Baru saja</p>
                                            </div>
                                        </div>
                                        <div class="timeline-item">
                                            <div class="timeline-marker bg-info"></div>
                                            <div class="timeline-content">
                                                <small class="text-muted">Auto refresh</small>
                                                <p class="mb-0 fw-medium">5 menit lagi</p>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        @else
            <!-- Gambar tidak ditemukan dengan UI menarik -->
            <div class="error-state text-center py-5">
                <div class="error-icon mb-4">
                    <i class="fas fa-image-slash fa-5x text-warning
                        animate-pulse" style="animation-duration: 2s;"></i>
                </div>
                <div class="error-content mx-auto" style="max-width: 500px;">
                    <h3 class="fw-bold text-warning mb-3">Gambar Tidak Tersedia</h3>
                    <p class="text-muted mb-4">
                        File gambar tidak ditemukan di server.
                        Silakan hubungi administrator untuk memperbarui jadwal.
                    </p>
                    <div class="action-buttons">
                        <button class="btn btn-primary me-2" onclick="location.reload()">
                            <i class="fas fa-sync-alt me-2"></i> Refresh Halaman
                        </button>
                        <button class="btn btn-outline-primary">
                            <i class="fas fa-headset me-2"></i> Hubungi Admin
                        </button>
                    </div>
                </div>
            </div>
        @endif
    @else
        <!-- Tidak ada jadwal dengan UI menarik -->
        <div class="empty-state text-center py-5">
            <div class="empty-icon mb-4">
                <div class="calendar-empty">
                    <i class="fas fa-calendar-times fa-5x text-muted"></i>
                    <div class="pulse-ring"></div>
                </div>
            </div>
            <div class="empty-content mx-auto" style="max-width: 500px;">
                <h3 class="fw-bold text-muted mb-3">Belum Ada Jadwal</h3>
                <p class="text-muted mb-4">
                    Jadwal untuk <span class="fw-bold">{{ $currentMonth }} {{ $currentYear }}</span>
                    belum tersedia atau belum diaktifkan oleh administrator.
                </p>
                <div class="countdown-timer mb-4">
                    <div class="timer-label small text-muted mb-2">
                        <i class="fas fa-clock me-1"></i>
                        Halaman akan refresh otomatis dalam:
                    </div>
                    <div class="timer-display">
                        <span class="minutes">05</span>:<span class="seconds">00</span>
                    </div>
                </div>
                <div class="info-box bg-primary-light rounded p-3">
                    <i class="fas fa-lightbulb text-primary me-2"></i>
                    <span class="small">Hubungi administrator untuk membuat jadwal bulan ini</span>
                </div>
            </div>
        </div>
    @endif
</div>
@endsection

@section('mobile-content')
<!-- MOBILE - TAMPILAN MENARIK -->
<div class="mobile-container">
    <!-- Mobile Header -->
    <div class="mobile-header-jadwal">
        <div class="d-flex align-items-center justify-content-between mb-3">
            <div>
                <h5 class="fw-bold mb-1 text-white">
                    <i class="fas fa-calendar-alt me-2"></i>
                    Jadwal MOD {{ $currentMonth }}
                </h5>
                <p class="text-white-75 small mb-0">{{ $currentYear }}</p>
            </div>
            <div class="status-badge">
                <span class="badge bg-{{ $jadwal ? 'success' : 'secondary' }} rounded-pill">
                    {{ $jadwal ? '✓ Tersedia' : '○ Menunggu' }}
                </span>
            </div>
        </div>

        <!-- Refresh Button -->
        <div class="refresh-mobile">
            <button class="btn btn-sm btn-light rounded-pill" onclick="location.reload()">
                <i class="fas fa-sync-alt me-1"></i> Refresh
            </button>
        </div>
    </div>

    @if($jadwal && $jadwal->image_path)
        @php
            $imageFullPath = storage_path('app/public/' . $jadwal->image_path);
            $imageExists = file_exists($imageFullPath);
            $imageUrl = $imageExists ? asset('storage/' . $jadwal->image_path) : null;
        @endphp

        @if($imageExists && $imageUrl)
            <!-- Mobile Jadwal Container -->
            <div class="mobile-jadwal-container">
                <!-- Jadwal Info -->
                <div class="card border-0 shadow-sm mb-3 mobile-info-card">
                    <div class="card-body p-3">
                        <h6 class="fw-bold mb-2 text-truncate">
                            <i class="fas fa-calendar-day me-2 text-primary"></i>
                            {{ $jadwal->title }}
                        </h6>
                        <div class="d-flex flex-wrap gap-2">
                            <span class="badge bg-primary rounded-pill small">
                                {{ $currentMonth }}
                            </span>
                            <span class="badge bg-accent-blue rounded-pill small">
                                {{ $currentYear }}
                            </span>
                            <span class="badge bg-success rounded-pill small">
                                Aktif
                            </span>
                        </div>
                    </div>
                </div>

                <!-- Gambar Jadwal Mobile -->
                <div class="mobile-image-container">
                    <div class="image-wrapper-mobile" id="mobileImageContainer">
                        <div class="image-inner-wrapper" id="mobileImageWrapper">
                            <img src="{{ $imageUrl }}"
                                 alt="Jadwal {{ $currentMonth }} {{ $currentYear }}"
                                 class="img-fluid rounded shadow"
                                 id="jadwalImageMobile"
                                 data-original-src="{{ $imageUrl }}">
                        </div>

                        <!-- Boundary Indicator Mobile -->
                        <div class="boundary-indicator-mobile" id="mobileBoundaryIndicator">
                            <div class="boundary-line top"></div>
                            <div class="boundary-line bottom"></div>
                            <div class="boundary-line left"></div>
                            <div class="boundary-line right"></div>
                        </div>

                        <!-- Zoom Controls Mobile -->
                        <div class="mobile-zoom-controls" id="mobileZoomControls">
                            <div class="controls-wrapper">
                                <button class="btn btn-sm btn-light rounded-circle shadow" onclick="mobileZoomIn()">
                                    <i class="fas fa-plus"></i>
                                </button>
                                <div class="zoom-level" id="mobileZoomIndicator">
                                    100%
                                </div>
                                <button class="btn btn-sm btn-light rounded-circle shadow" onclick="mobileZoomOut()">
                                    <i class="fas fa-minus"></i>
                                </button>
                                <button class="btn btn-sm btn-light rounded-circle shadow" onclick="mobileResetZoom()">
                                    <i class="fas fa-undo"></i>
                                </button>
                            </div>
                        </div>
                    </div>

                    <!-- Instructions -->
                    <div class="mobile-instructions mt-3">
                        <div class="instruction-item">
                            <i class="fas fa-hand-pointer text-primary me-2"></i>
                            <span class="small">Pinch untuk zoom</span>
                        </div>
                        <div class="instruction-item">
                            <i class="fas fa-hand-rock text-success me-2"></i>
                            <span class="small">Drag untuk geser</span>
                        </div>
                        <div class="instruction-item">
                            <i class="fas fa-clock text-warning me-2"></i>
                            <span class="small">Auto refresh 5 menit</span>
                        </div>
                    </div>
                </div>

                <!-- Quick Info -->
                <div class="mobile-quick-info mt-3">
                    <div class="info-grid">
                        <div class="info-item">
                            <div class="info-icon bg-primary">
                                <i class="fas fa-calendar-plus"></i>
                            </div>
                            <div class="info-content">
                                <small class="text-muted">Dibuat</small>
                                <p class="mb-0 fw-medium">{{ $jadwal->created_at->format('d M') }}</p>
                            </div>
                        </div>
                        <div class="info-item">
                            <div class="info-icon bg-success">
                                <i class="fas fa-eye"></i>
                            </div>
                            <div class="info-content">
                                <small class="text-muted">Status</small>
                                <p class="mb-0 fw-medium">Aktif</p>
                            </div>
                        </div>
                        <div class="info-item">
                            <div class="info-icon bg-info">
                                <i class="fas fa-sync-alt"></i>
                            </div>
                            <div class="info-content">
                                <small class="text-muted">Refresh</small>
                                <p class="mb-0 fw-medium">5 menit</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        @else
            <!-- Gambar tidak ditemukan (mobile) -->
            <div class="mobile-error-state">
                <div class="text-center py-4">
                    <i class="fas fa-image-slash fa-3x text-warning mb-3"></i>
                    <h6 class="text-warning mb-2">Gambar Tidak Tersedia</h6>
                    <p class="text-muted small mb-3">
                        File gambar tidak ditemukan
                    </p>
                    <button class="btn btn-sm btn-warning" onclick="location.reload()">
                        <i class="fas fa-redo me-1"></i> Coba Lagi
                    </button>
                </div>
            </div>
        @endif
    @else
        <!-- Tidak ada jadwal (mobile) -->
        <div class="mobile-empty-state">
            <div class="text-center py-5">
                <div class="empty-icon-mobile mb-3">
                    <i class="fas fa-calendar-times fa-3x text-muted"></i>
                </div>
                <h6 class="text-muted mb-2">Belum Ada Jadwal</h6>
                <p class="text-muted small mb-4">
                    {{ $currentMonth }} {{ $currentYear }}
                </p>

                <!-- Countdown Timer Mobile -->
                <div class="mobile-timer mb-4">
                    <div class="timer-circle">
                        <div class="timer-progress"></div>
                        <div class="timer-text">5:00</div>
                    </div>
                    <small class="text-muted d-block mt-2">Auto refresh</small>
                </div>

                <div class="alert alert-info small">
                    <i class="fas fa-info-circle me-1"></i>
                    Hubungi admin untuk membuat jadwal
                </div>
            </div>
        </div>
    @endif
</div>
@endsection

@push('styles')
<style>
/* ========== ANIMATIONS ========== */
@keyframes calendarPulse {
    0% { transform: scale(1); opacity: 1; }
    100% { transform: scale(1.1); opacity: 0.8; }
}

@keyframes slideUp {
    from { transform: translateY(20px); opacity: 0; }
    to { transform: translateY(0); opacity: 1; }
}

@keyframes fadeIn {
    from { opacity: 0; }
    to { opacity: 1; }
}

@keyframes pulse {
    0%, 100% { opacity: 1; }
    50% { opacity: 0.7; }
}

@keyframes shimmer {
    0% { background-position: -200px 0; }
    100% { background-position: 200px 0; }
}

@keyframes boundaryFlash {
    0%, 100% { opacity: 0; }
    50% { opacity: 1; }
}

/* ========== DESKTOP STYLES ========== */
.animated-header {
    animation: slideUp 0.6s ease-out;
}

.calendar-icon {
    position: relative;
}

.calendar-icon::after {
    content: '';
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    width: 80px;
    height: 80px;
    border-radius: 50%;
    background: var(--accent-blue-lightest);
    z-index: -1;
    animation: pulse 2s infinite;
}

.gradient-text {
    background: linear-gradient(90deg, var(--primary-orange), var(--accent-blue));
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.status-indicator {
    animation: fadeIn 1s ease-out;
    transition: all 0.3s ease;
}

.status-indicator:hover {
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

.status-dot {
    width: 8px;
    height: 8px;
    border-radius: 50%;
    animation: pulse 1.5s infinite;
}

.jadwal-container {
    animation: fadeIn 0.8s ease-out;
}

.animate-slide-up {
    animation: slideUp 0.5s ease-out;
}

.animate-fade-in {
    animation: fadeIn 0.6s ease-out;
}

.animate-pulse {
    animation: pulse 2s infinite;
}

.icon-wrapper {
    transition: transform 0.3s ease;
}

.icon-wrapper:hover {
    transform: rotate(15deg) scale(1.1);
}

.text-gradient {
    background: linear-gradient(90deg, var(--gray-800), var(--accent-blue));
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.image-frame {
    position: relative;
    border: 2px solid var(--gray-100);
    transition: all 0.3s ease;
    overflow: hidden !important;
}

.image-frame:hover {
    border-color: var(--accent-blue);
    box-shadow: var(--shadow-xl) !important;
}

.image-wrapper {
    position: relative;
    width: 100%;
    height: 100%;
    overflow: hidden;
    transform-origin: center center;
    transition: transform 0.3s cubic-bezier(0.25, 0.46, 0.45, 0.94);
}

#jadwalImage {
    display: block;
    max-width: none;
    transition: transform 0.3s cubic-bezier(0.25, 0.46, 0.45, 0.94);
    cursor: grab;
    user-select: none;
    touch-action: none;
}

#jadwalImage:active {
    cursor: grabbing;
}

.image-overlay {
    position: absolute;
    bottom: 0;
    left: 0;
    right: 0;
    background: linear-gradient(transparent, rgba(0,0,0,0.7));
    padding: 20px;
    opacity: 0;
    transition: opacity 0.3s ease;
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 10px;
    z-index: 10;
}

.image-frame:hover .image-overlay {
    opacity: 1;
}

.zoom-controls {
    display: flex;
    gap: 15px;
}

.zoom-indicator {
    background: rgba(255, 255, 255, 0.9);
    padding: 5px 15px;
    border-radius: 20px;
    font-weight: 600;
    color: var(--gray-800);
    box-shadow: var(--shadow-sm);
}

.image-loading {
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(255, 255, 255, 0.9);
    display: flex;
    align-items: center;
    justify-content: center;
    opacity: 0;
    transition: opacity 0.3s ease;
    z-index: 5;
}

.boundary-indicator {
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    pointer-events: none;
    z-index: 3;
}

.boundary-line {
    position: absolute;
    background: rgba(255, 0, 0, 0.3);
    display: none;
}

.boundary-line.top {
    top: 0;
    left: 0;
    right: 0;
    height: 2px;
}

.boundary-line.bottom {
    bottom: 0;
    left: 0;
    right: 0;
    height: 2px;
}

.boundary-line.left {
    left: 0;
    top: 0;
    bottom: 0;
    width: 2px;
}

.boundary-line.right {
    right: 0;
    top: 0;
    bottom: 0;
    width: 2px;
}

.info-card {
    transition: all 0.3s ease;
    cursor: default;
}

.info-card:hover {
    transform: translateY(-2px);
    box-shadow: var(--shadow-sm);
}

.bg-primary-light { background-color: rgba(var(--primary-orange-rgb), 0.1); }
.bg-success-light { background-color: rgba(var(--success-rgb), 0.1); }
.bg-warning-light { background-color: rgba(var(--warning-rgb), 0.1); }

.timeline {
    position: relative;
    padding-left: 20px;
}

.timeline::before {
    content: '';
    position: absolute;
    left: 0;
    top: 0;
    bottom: 0;
    width: 2px;
    background: var(--gray-200);
}

.timeline-item {
    position: relative;
    margin-bottom: 15px;
}

.timeline-marker {
    position: absolute;
    left: -26px;
    top: 5px;
    width: 12px;
    height: 12px;
    border-radius: 50%;
    border: 2px solid white;
    box-shadow: var(--shadow-sm);
}

.timeline-content {
    padding-left: 10px;
}

.error-state, .empty-state {
    animation: fadeIn 0.8s ease-out;
}

.calendar-empty {
    position: relative;
    display: inline-block;
}

.pulse-ring {
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    width: 100px;
    height: 100px;
    border: 3px solid var(--gray-300);
    border-radius: 50%;
    animation: pulse 2s infinite;
}

.countdown-timer {
    display: inline-block;
    padding: 10px 20px;
    background: var(--gray-100);
    border-radius: var(--radius-lg);
}

.timer-display {
    font-family: monospace;
    font-size: 1.5rem;
    font-weight: bold;
    color: var(--primary-orange);
}

/* ========== MOBILE STYLES ========== */
.mobile-header-jadwal {
    background: linear-gradient(135deg, var(--navy) 0%, var(--navy-medium) 100%);
    padding: 20px;
    border-radius: var(--radius-lg);
    margin-bottom: 20px;
    color: white;
    position: sticky;
    top: 0;
    z-index: 20;
}

.mobile-info-card {
    transition: transform 0.3s ease;
}

.mobile-info-card:hover {
    transform: translateY(-2px);
}

.mobile-image-container {
    position: relative;
}

.image-wrapper-mobile {
    position: relative;
    border-radius: var(--radius-lg);
    overflow: hidden;
    border: 1px solid var(--gray-200);
    height: 400px;
    overflow: hidden;
}

.image-inner-wrapper {
    position: relative;
    width: 100%;
    height: 100%;
    overflow: hidden;
    transform-origin: center center;
    transition: transform 0.3s cubic-bezier(0.25, 0.46, 0.45, 0.94);
}

#jadwalImageMobile {
    display: block;
    max-width: none;
    height: 100%;
    width: 100%;
    object-fit: contain;
    transition: transform 0.3s cubic-bezier(0.25, 0.46, 0.45, 0.94);
    cursor: grab;
    user-select: none;
    touch-action: none;
}

#jadwalImageMobile:active {
    cursor: grabbing;
}

.boundary-indicator-mobile {
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    pointer-events: none;
    z-index: 3;
}

.mobile-zoom-controls {
    position: absolute;
    bottom: 10px;
    left: 0;
    right: 0;
    display: flex;
    justify-content: center;
    z-index: 10;
}

.controls-wrapper {
    display: flex;
    align-items: center;
    gap: 10px;
    background: rgba(255, 255, 255, 0.95);
    padding: 8px 15px;
    border-radius: 25px;
    backdrop-filter: blur(10px);
    box-shadow: 0 4px 15px rgba(0,0,0,0.1);
}

.zoom-level {
    font-weight: 600;
    color: var(--gray-800);
    min-width: 50px;
    text-align: center;
    font-size: 0.9rem;
}

.mobile-instructions {
    display: flex;
    justify-content: space-around;
    padding: 10px;
    background: var(--gray-50);
    border-radius: var(--radius-lg);
    margin-top: 15px;
}

.instruction-item {
    display: flex;
    align-items: center;
    font-size: 0.75rem;
}

.mobile-quick-info {
    padding: 15px;
    background: var(--gray-50);
    border-radius: var(--radius-lg);
    margin-top: 15px;
}

.info-grid {
    display: grid;
    grid-template-columns: repeat(3, 1fr);
    gap: 10px;
}

.info-item {
    text-align: center;
}

.info-icon {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    margin: 0 auto 5px;
    color: white;
}

.info-content small {
    font-size: 0.7rem;
}

.mobile-timer {
    display: flex;
    flex-direction: column;
    align-items: center;
}

.timer-circle {
    position: relative;
    width: 80px;
    height: 80px;
}

.timer-progress {
    position: absolute;
    width: 100%;
    height: 100%;
    border: 3px solid var(--accent-blue);
    border-radius: 50%;
    border-right-color: transparent;
    animation: rotate 60s linear infinite;
}

.timer-text {
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    font-weight: bold;
    color: var(--accent-blue);
}

@keyframes rotate {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

/* ========== RESPONSIVE ADJUSTMENTS ========== */
@media (max-width: 768px) {
    .image-overlay {
        opacity: 1;
        background: rgba(0,0,0,0.5);
    }

    .zoom-controls button {
        width: 40px;
        height: 40px;
        font-size: 0.9rem;
    }
}

/* ========== ZOOM EFFECTS ========== */
.zoom-transition {
    transition: transform 0.3s cubic-bezier(0.25, 0.46, 0.45, 0.94);
}

.boundary-warning {
    animation: boundaryFlash 0.3s ease-in-out 2;
}
</style>
@endpush

@section('scripts')
<script>
// ========== GLOBAL ZOOM MANAGER ==========
class ZoomManager {
    constructor(imageId, containerId, wrapperId) {
        this.image = document.getElementById(imageId);
        this.container = document.getElementById(containerId);
        this.wrapper = document.getElementById(wrapperId);
        this.scale = 1;
        this.minScale = 1;
        this.maxScale = 3;
        this.translateX = 0;
        this.translateY = 0;
        this.isDragging = false;
        this.lastX = 0;
        this.lastY = 0;
        this.velocityX = 0;
        this.velocityY = 0;
        this.lastTime = 0;
        this.boundaryIndicator = null;
        this.isMobile = false;
        
        this.init();
    }
    
    init() {
        if (!this.image || !this.container || !this.wrapper) return;
        
        this.setupBoundaryIndicator();
        this.setupEventListeners();
        this.updateTransform();
    }
    
    setupBoundaryIndicator() {
        this.boundaryIndicator = document.getElementById(this.container.id === 'imageContainer' ? 'boundaryIndicator' : 'mobileBoundaryIndicator');
    }
    
    setupEventListeners() {
        // Mouse events for desktop
        if (!this.isMobile) {
            this.image.addEventListener('wheel', (e) => this.handleWheel(e));
            this.image.addEventListener('mousedown', (e) => this.startDrag(e));
            document.addEventListener('mousemove', (e) => this.drag(e));
            document.addEventListener('mouseup', () => this.endDrag());
            this.image.addEventListener('dblclick', (e) => this.handleDoubleClick(e));
        }
        
        // Touch events for mobile
        this.image.addEventListener('touchstart', (e) => this.handleTouchStart(e), { passive: false });
        this.image.addEventListener('touchmove', (e) => this.handleTouchMove(e), { passive: false });
        this.image.addEventListener('touchend', () => this.handleTouchEnd());
        
        // Prevent default touch actions
        this.image.addEventListener('touchmove', (e) => {
            if (this.scale > 1) {
                e.preventDefault();
            }
        }, { passive: false });
    }
    
    handleWheel(e) {
        e.preventDefault();
        
        const rect = this.container.getBoundingClientRect();
        const mouseX = e.clientX - rect.left;
        const mouseY = e.clientY - rect.top;
        
        const delta = e.deltaY > 0 ? -0.1 : 0.1;
        const oldScale = this.scale;
        
        this.scale = Math.max(this.minScale, Math.min(this.maxScale, this.scale + delta));
        
        if (this.scale !== oldScale) {
            // Calculate new translation to zoom toward mouse position
            const scaleDiff = this.scale - oldScale;
            this.translateX -= (mouseX - rect.width / 2 - this.translateX) * (scaleDiff / oldScale);
            this.translateY -= (mouseY - rect.height / 2 - this.translateY) * (scaleDiff / oldScale);
            
            this.applyBoundaries();
            this.updateTransform();
            this.updateZoomIndicator();
        }
    }
    
    handleDoubleClick(e) {
        e.preventDefault();
        
        if (this.scale === 1) {
            const rect = this.container.getBoundingClientRect();
            const mouseX = e.clientX - rect.left;
            const mouseY = e.clientY - rect.top;
            
            this.scale = 2;
            this.translateX = (rect.width / 2 - mouseX) * 1;
            this.translateY = (rect.height / 2 - mouseY) * 1;
        } else {
            this.scale = 1;
            this.translateX = 0;
            this.translateY = 0;
        }
        
        this.applyBoundaries();
        this.updateTransform();
        this.updateZoomIndicator();
    }
    
    startDrag(e) {
        if (this.scale <= 1) return;
        
        this.isDragging = true;
        this.lastX = e.clientX;
        this.lastY = e.clientY;
        this.lastTime = Date.now();
        this.velocityX = 0;
        this.velocityY = 0;
        this.image.style.cursor = 'grabbing';
    }
    
    drag(e) {
        if (!this.isDragging) return;
        
        const currentTime = Date.now();
        const deltaTime = currentTime - this.lastTime;
        
        if (deltaTime > 0) {
            const deltaX = e.clientX - this.lastX;
            const deltaY = e.clientY - this.lastY;
            
            this.velocityX = deltaX / deltaTime;
            this.velocityY = deltaY / deltaTime;
            
            this.translateX += deltaX;
            this.translateY += deltaY;
            
            this.applyBoundaries();
            this.updateTransform();
            
            this.lastX = e.clientX;
            this.lastY = e.clientY;
            this.lastTime = currentTime;
        }
    }
    
    endDrag() {
        if (!this.isDragging) return;
        
        this.isDragging = false;
        this.image.style.cursor = 'grab';
        
        // Apply inertia
        this.applyInertia();
    }
    
    applyInertia() {
        const inertiaStrength = 0.9;
        const minVelocity = 0.1;
        
        const applyInertiaFrame = () => {
            if (Math.abs(this.velocityX) < minVelocity && Math.abs(this.velocityY) < minVelocity) {
                this.velocityX = 0;
                this.velocityY = 0;
                return;
            }
            
            this.translateX += this.velocityX * 16;
            this.translateY += this.velocityY * 16;
            
            this.velocityX *= inertiaStrength;
            this.velocityY *= inertiaStrength;
            
            this.applyBoundaries();
            this.updateTransform();
            
            requestAnimationFrame(applyInertiaFrame);
        };
        
        requestAnimationFrame(applyInertiaFrame);
    }
    
    handleTouchStart(e) {
        if (e.touches.length === 1) {
            this.startDrag({
                clientX: e.touches[0].clientX,
                clientY: e.touches[0].clientY
            });
        } else if (e.touches.length === 2) {
            this.handlePinchStart(e);
        }
    }
    
    handleTouchMove(e) {
        if (e.touches.length === 1 && this.isDragging) {
            this.drag({
                clientX: e.touches[0].clientX,
                clientY: e.touches[0].clientY
            });
        } else if (e.touches.length === 2) {
            this.handlePinchMove(e);
        }
    }
    
    handleTouchEnd() {
        this.endDrag();
    }
    
    handlePinchStart(e) {
        const touch1 = e.touches[0];
        const touch2 = e.touches[1];
        
        this.pinchStartDistance = Math.hypot(
            touch2.clientX - touch1.clientX,
            touch2.clientY - touch1.clientY
        );
        this.pinchStartScale = this.scale;
        this.pinchCenterX = (touch1.clientX + touch2.clientX) / 2;
        this.pinchCenterY = (touch1.clientY + touch2.clientY) / 2;
    }
    
    handlePinchMove(e) {
        e.preventDefault();
        
        const touch1 = e.touches[0];
        const touch2 = e.touches[1];
        
        const currentDistance = Math.hypot(
            touch2.clientX - touch1.clientX,
            touch2.clientY - touch1.clientY
        );
        
        if (this.pinchStartDistance) {
            const scaleChange = currentDistance / this.pinchStartDistance;
            this.scale = Math.max(this.minScale, Math.min(this.maxScale, this.pinchStartScale * scaleChange));
            
            // Adjust translation based on pinch center
            const rect = this.container.getBoundingClientRect();
            const centerX = this.pinchCenterX - rect.left;
            const centerY = this.pinchCenterY - rect.top;
            
            const scaleDiff = this.scale - this.pinchStartScale;
            this.translateX -= (centerX - rect.width / 2 - this.translateX) * (scaleDiff / this.pinchStartScale);
            this.translateY -= (centerY - rect.height / 2 - this.translateY) * (scaleDiff / this.pinchStartScale);
            
            this.applyBoundaries();
            this.updateTransform();
            this.updateZoomIndicator();
        }
    }
    
    applyBoundaries() {
        const rect = this.container.getBoundingClientRect();
        const imageRect = this.image.getBoundingClientRect();
        
        const scaledWidth = imageRect.width * this.scale;
        const scaledHeight = imageRect.height * this.scale;
        
        const maxX = Math.max(0, (scaledWidth - rect.width) / 2);
        const maxY = Math.max(0, (scaledHeight - rect.height) / 2);
        
        if (scaledWidth <= rect.width) {
            this.translateX = 0;
        } else {
            this.translateX = Math.max(-maxX, Math.min(maxX, this.translateX));
        }
        
        if (scaledHeight <= rect.height) {
            this.translateY = 0;
        } else {
            this.translateY = Math.max(-maxY, Math.min(maxY, this.translateY));
        }
        
        // Show boundary warning if at edge
        this.showBoundaryWarning();
    }
    
    showBoundaryWarning() {
        if (!this.boundaryIndicator) return;
        
        const rect = this.container.getBoundingClientRect();
        const imageRect = this.image.getBoundingClientRect();
        
        const scaledWidth = imageRect.width * this.scale;
        const scaledHeight = imageRect.height * this.scale;
        
        const maxX = Math.max(0, (scaledWidth - rect.width) / 2);
        const maxY = Math.max(0, (scaledHeight - rect.height) / 2);
        
        const lines = this.boundaryIndicator.querySelectorAll('.boundary-line');
        
        // Hide all lines first
        lines.forEach(line => line.style.display = 'none');
        
        // Show boundary lines when at edge
        if (Math.abs(this.translateX) >= maxX - 1) {
            if (this.translateX > 0) {
                lines[3].style.display = 'block'; // right
            } else {
                lines[2].style.display = 'block'; // left
            }
        }
        
        if (Math.abs(this.translateY) >= maxY - 1) {
            if (this.translateY > 0) {
                lines[1].style.display = 'block'; // bottom
            } else {
                lines[0].style.display = 'block'; // top
            }
        }
    }
    
    updateTransform() {
        if (this.wrapper) {
            this.wrapper.style.transform = `translate(${this.translateX}px, ${this.translateY}px) scale(${this.scale})`;
        } else if (this.image) {
            this.image.style.transform = `translate(${this.translateX}px, ${this.translateY}px) scale(${this.scale})`;
        }
    }
    
    updateZoomIndicator() {
        const indicator = document.getElementById(this.container.id === 'imageContainer' ? 'zoomIndicator' : 'mobileZoomIndicator');
        if (indicator) {
            indicator.textContent = Math.round(this.scale * 100) + '%';
        }
    }
    
    zoomIn() {
        const oldScale = this.scale;
        this.scale = Math.min(this.maxScale, this.scale + 0.5);
        
        if (this.scale !== oldScale) {
            this.applyBoundaries();
            this.updateTransform();
            this.updateZoomIndicator();
        }
    }
    
    zoomOut() {
        const oldScale = this.scale;
        this.scale = Math.max(this.minScale, this.scale - 0.5);
        
        if (this.scale !== oldScale) {
            if (this.scale === 1) {
                this.translateX = 0;
                this.translateY = 0;
            }
            this.applyBoundaries();
            this.updateTransform();
            this.updateZoomIndicator();
        }
    }
    
    resetZoom() {
        this.scale = 1;
        this.translateX = 0;
        this.translateY = 0;
        this.applyBoundaries();
        this.updateTransform();
        this.updateZoomIndicator();
    }
}

// ========== DESKTOP ZOOM MANAGER ==========
let desktopZoomManager = null;

// ========== MOBILE ZOOM MANAGER ==========
let mobileZoomManager = null;

// ========== GLOBAL FUNCTIONS ==========
function zoomIn() {
    if (desktopZoomManager) {
        desktopZoomManager.zoomIn();
    }
}

function zoomOut() {
    if (desktopZoomManager) {
        desktopZoomManager.zoomOut();
    }
}

function resetZoom() {
    if (desktopZoomManager) {
        desktopZoomManager.resetZoom();
    }
}

function mobileZoomIn() {
    if (mobileZoomManager) {
        mobileZoomManager.zoomIn();
    }
}

function mobileZoomOut() {
    if (mobileZoomManager) {
        mobileZoomManager.zoomOut();
    }
}

function mobileResetZoom() {
    if (mobileZoomManager) {
        mobileZoomManager.resetZoom();
    }
}

function toggleFullscreen() {
    const container = document.getElementById('imageContainer');
    if (!container) return;
    
    if (!document.fullscreenElement) {
        if (container.requestFullscreen) {
            container.requestFullscreen();
        } else if (container.webkitRequestFullscreen) {
            container.webkitRequestFullscreen();
        } else if (container.msRequestFullscreen) {
            container.msRequestFullscreen();
        }
    } else {
        if (document.exitFullscreen) {
            document.exitFullscreen();
        } else if (document.webkitExitFullscreen) {
            document.webkitExitFullscreen();
        } else if (document.msExitFullscreen) {
            document.msExitFullscreen();
        }
    }
}

// Countdown timer
function startCountdownTimer() {
    let minutes = 4;
    let seconds = 59;

    const timer = setInterval(function() {
        // Update desktop timer
        const minutesEl = document.querySelector('.minutes');
        const secondsEl = document.querySelector('.seconds');

        if (minutesEl && secondsEl) {
            minutesEl.textContent = minutes.toString().padStart(2, '0');
            secondsEl.textContent = seconds.toString().padStart(2, '0');
        }

        // Update mobile timer
        const timerText = document.querySelector('.timer-text');
        if (timerText) {
            timerText.textContent = `${minutes}:${seconds.toString().padStart(2, '0')}`;
        }

        if (seconds === 0) {
            if (minutes === 0) {
                clearInterval(timer);
                location.reload();
            } else {
                minutes--;
                seconds = 59;
            }
        } else {
            seconds--;
        }
    }, 1000);
}

// Loading animation
window.addEventListener('load', function() {
    const loading = document.getElementById('imageLoading');
    if (loading) {
        setTimeout(() => {
            loading.style.opacity = '0';
            setTimeout(() => {
                loading.style.display = 'none';
            }, 300);
        }, 500);
    }
});

// Initialize when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    // Initialize desktop zoom manager
    desktopZoomManager = new ZoomManager('jadwalImage', 'imageContainer', 'imageWrapper');
    
    // Initialize mobile zoom manager
    mobileZoomManager = new ZoomManager('jadwalImageMobile', 'mobileImageContainer', 'mobileImageWrapper');
    if (mobileZoomManager) mobileZoomManager.isMobile = true;
    
    // Start countdown timer
    startCountdownTimer();
    
    // Auto refresh every 5 minutes
    setInterval(function() {
        location.reload();
    }, 5 * 60 * 1000);
    
    // Handle fullscreen change
    document.addEventListener('fullscreenchange', handleFullscreenChange);
    document.addEventListener('webkitfullscreenchange', handleFullscreenChange);
    document.addEventListener('msfullscreenchange', handleFullscreenChange);
    
    function handleFullscreenChange() {
        if (desktopZoomManager) {
            // Recalculate boundaries when fullscreen changes
            setTimeout(() => {
                desktopZoomManager.applyBoundaries();
                desktopZoomManager.updateTransform();
            }, 100);
        }
    }
});
</script>
@endsection