@extends('layouts.AdminLayout')

@php
    $today = now();
    $year = $today->year;
    $month = $today->month;
    $monthName = $today->format('F'); // Already in English
    
    $daysInMonth = cal_days_in_month(CAL_GREGORIAN, $month, $year);
    $firstDay = $today->copy()->firstOfMonth()->dayOfWeek; // 0=Sunday, 1=Monday, ...
    
    // Day names in English
    $dayNames = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];
@endphp

@section('content')
<!-- DESKTOP -->
<div class="d-none d-lg-block">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h2 class="fw-bold">Manage Schedule</h2>
        <div class="d-flex gap-3">
            <button class="btn btn-outline" data-bs-toggle="modal" data-bs-target="#deleteAllModal">
                <i class="fas fa-trash me-1"></i> Delete All
            </button>
            <div class="btn-group">
                <button class="btn bg-primary text-white dropdown-toggle" type="button" data-bs-toggle="dropdown" aria-expanded="false">
                    <i class="fas fa-download me-1"></i> Save
                </button>
                <ul class="dropdown-menu">
                    <li><a class="dropdown-item" href="#" onclick="window.simpanLokal()"><i class="fas fa-download me-2"></i> Download to Device</a></li>
                    <li><a class="dropdown-item" href="#" data-bs-toggle="modal" data-bs-target="#uploadModal" onclick="window.prepareUpload()"><i class="fas fa-cloud-upload-alt me-2"></i> Save to Server</a></li>
                </ul>
            </div>
        </div>
    </div>

    <div class="mb-4">
        <label for="calendar-title" class="form-label fw-medium">Schedule Title</label>
        <input type="text"
               id="calendar-title"
               class="form-control"
               placeholder="Example: MOD Schedule {{ $monthName }} {{ $year }}"
               value="MOD Schedule {{ $monthName }} {{ $year }}">
    </div>

    <div class="bg-white rounded-3 p-4 shadow-sm" style="border: 1px solid var(--gray-200);">
        <div class="d-flex justify-content-center align-items-center mb-4">
            <button class="btn btn-sm btn-outline-secondary rounded-circle me-3" style="width: 40px; height: 40px;" onclick="window.changeMonth(-1)">
                <i class="fas fa-chevron-left"></i>
            </button>
            <h3 class="fw-semibold m-0" id="month-year">{{ $monthName }} {{ $year }}</h3>
            <button class="btn btn-sm btn-outline-secondary rounded-circle ms-3" style="width: 40px; height: 40px;" onclick="window.changeMonth(1)">
                <i class="fas fa-chevron-right"></i>
            </button>
        </div>

        <div id="schedule-container">
            <div id="calendar-title-display" class="text-center fw-bold mb-3" style="font-size: 1.3rem; color: var(--gray-800);">MOD Schedule {{ $monthName }} {{ $year }}</div>
            <table class="table w-100" style="border-collapse: collapse;">
                <thead>
                    <tr class="text-center" style="color: var(--gray-700); font-weight: 600;">
                        @foreach($dayNames as $day)
                            <th class="py-3">{{ $day }}</th>
                        @endforeach
                    </tr>
                </thead>
                <tbody id="calendar-body">
                    @php
                        $date = 1;
                        for ($week = 0; $week < 6; $week++):
                            if ($date > $daysInMonth) break;
                            echo '<tr>';
                            for ($dayIndex = 0; $dayIndex < 7; $dayIndex++):
                                echo '<td style="padding:0.4rem; border:1px solid var(--gray-200); text-align:left; height:95px;">';
                                if ($week === 0 && $dayIndex < $firstDay):
                                    // empty
                                elseif ($date <= $daysInMonth):
                                    echo '<div class="date-box" style="font-weight:600; margin-bottom:0.25rem;">' . $date . '</div>';
                                    echo '<div class="event-placeholder" data-date="' . $date . '"></div>';
                                    $date++;
                                endif;
                                echo '</td>';
                            endfor;
                            echo '</tr>';
                        endfor;
                    @endphp
                </tbody>
            </table>
        </div>
    </div>

    <div class="mt-5">
        <div class="d-flex justify-content-between align-items-center mb-3">
            <h3 class="fw-bold fs-5">Saved Schedules</h3>
            <button class="btn btn-sm btn-outline-primary" onclick="window.muatGaleri()">
                <i class="fas fa-sync me-1"></i> Refresh
            </button>
        </div>
        <div id="gallery-container" class="row g-3">
            <div class="col-12 text-center py-4">
                <div class="spinner-border text-primary mb-3" role="status">
                    <span class="visually-hidden">Loading...</span>
                </div>
                <p class="text-muted">Loading schedules...</p>
            </div>
        </div>
    </div>
</div>
@endsection

@section('mobile-content')
<!-- MOBILE -->
<div class="bg-white rounded-3 p-3 mb-4 shadow-sm" style="border: 1px solid var(--gray-200);">
    <div class="mb-3">
        <label for="calendar-title-mobile" class="form-label fw-medium fs-6">Schedule Title</label>
        <input type="text"
               id="calendar-title-mobile"
               class="form-control form-control-sm"
               placeholder="Example: MOD Schedule {{ $monthName }} {{ $year }}"
               value="MOD Schedule {{ $monthName }} {{ $year }}">
    </div>

    <div class="d-flex justify-content-between align-items-center mb-3">
        <button class="btn btn-sm btn-outline-secondary rounded-circle" style="width: 36px; height: 36px;" onclick="window.changeMonth(-1)">
            <i class="fas fa-chevron-left"></i>
        </button>
        <h4 class="fw-semibold m-0" id="month-year-mobile">{{ $monthName }} {{ $year }}</h4>
        <button class="btn btn-sm btn-outline-secondary rounded-circle" style="width: 36px; height: 36px;" onclick="window.changeMonth(1)">
            <i class="fas fa-chevron-right"></i>
        </button>
    </div>

    <!-- Hidden container for export - MODIFIED FOR DESKTOP-LIKE DISPLAY -->
    <div id="schedule-container-mobile" style="display: none; background: white; padding: 20px; border-radius: 8px;">
        <div id="calendar-title-display-mobile" class="text-center fw-bold mb-3" style="font-size: 1.3rem; color: var(--gray-800);">MOD Schedule {{ $monthName }} {{ $year }}</div>
        <table class="table w-100" style="border-collapse: collapse; min-width: 800px;">
            <thead>
                <tr class="text-center" style="color: var(--gray-700); font-weight: 600;">
                    @foreach($dayNames as $day)
                        <th class="py-3" style="width: 14.28%;">{{ $day }}</th>
                    @endforeach
                </tr>
            </thead>
            <tbody id="calendar-body-hidden">
                @php
                    $date = 1;
                    for ($week = 0; $week < 6; $week++):
                        if ($date > $daysInMonth) break;
                        echo '<tr>';
                        for ($dayIndex = 0; $dayIndex < 7; $dayIndex++):
                            echo '<td style="padding:0.4rem; border:1px solid var(--gray-200); text-align:left; height:95px; vertical-align:top;">';
                            if ($week === 0 && $dayIndex < $firstDay):
                                // empty
                            elseif ($date <= $daysInMonth):
                                $dayColor = '';
                                if ($dayIndex === 0) {
                                    $dayColor = 'color: var(--danger)!important;';
                                } elseif ($dayIndex === 6) {
                                    $dayColor = 'color: var(--info)!important;';
                                }
                                
                                echo '<div class="date-box" style="font-weight:600; margin-bottom:0.25rem; ' . $dayColor . '">' . $date . '</div>';
                                echo '<div class="event-placeholder-export" data-date="' . $date . '"></div>';
                                $date++;
                            endif;
                            echo '</td>';
                        endfor;
                        echo '</tr>';
                    endfor;
                @endphp
            </tbody>
        </table>
    </div>

    <!-- Responsive mobile calendar display - MODIFIED INITIALS TO ENGLISH -->
    <div class="mobile-calendar-responsive">
        <div class="calendar-grid">
            @php
                $date = 1;
                // Header with English initials: S M T W T F S
                echo '<div class="calendar-header">';
                foreach(['S', 'M', 'T', 'W', 'T', 'F', 'S'] as $day):
                    echo '<div class="calendar-day-header">' . $day . '</div>';
                endforeach;
                echo '</div>';
                
                for ($week = 0; $week < 6; $week++):
                    if ($date > $daysInMonth) break;
                    echo '<div class="calendar-week">';
                    for ($dayIndex = 0; $dayIndex < 7; $dayIndex++):
                        echo '<div class="calendar-day-cell">';
                        if ($week === 0 && $dayIndex < $firstDay):
                            echo '<div class="calendar-date empty"></div>';
                        elseif ($date <= $daysInMonth):
                            echo '<div class="calendar-date active">';
                            echo '<div class="date-number">' . $date . '</div>';
                            echo '<div class="event-placeholder" data-date="' . $date . '"></div>';
                            echo '</div>';
                            $date++;
                        else:
                            echo '<div class="calendar-date empty"></div>';
                        endif;
                        echo '</div>';
                    endfor;
                    echo '</div>';
                endfor;
            @endphp
        </div>
    </div>

    <div class="d-grid gap-2 mt-3">
        <button class="btn btn-outline" data-bs-toggle="modal" data-bs-target="#deleteAllModal">
            <i class="fas fa-trash me-1"></i> Delete All
        </button>
        <div class="btn-group">
            <button class="btn bg-primary text-white dropdown-toggle" type="button" data-bs-toggle="dropdown" aria-expanded="false">
                <i class="fas fa-download me-1"></i> Save
            </button>
            <ul class="dropdown-menu">
                <li><a class="dropdown-item" href="#" onclick="window.simpanLokal()"><i class="fas fa-download me-2"></i> Download</a></li>
                <li><a class="dropdown-item" href="#" data-bs-toggle="modal" data-bs-target="#uploadModal" onclick="window.prepareUpload()"><i class="fas fa-cloud-upload-alt me-2"></i> Save to Server</a></li>
            </ul>
        </div>
    </div>
</div>

<div class="mt-4">
    <div class="d-flex justify-content-between align-items-center mb-3">
        <h5 class="fw-bold">Saved Schedules</h5>
        <button class="btn btn-sm btn-outline-primary" onclick="window.muatGaleri()">
            <i class="fas fa-sync me-1"></i> Refresh
        </button>
    </div>
    <div id="gallery-container-mobile" class="row g-2">
        <div class="col-12 text-center py-3">
            <div class="spinner-border text-primary mb-3" role="status">
                <span class="visually-hidden">Loading...</span>
            </div>
            <p class="text-muted">Loading schedules...</p>
        </div>
    </div>
</div>

<style>
/* STYLING FOR RESPONSIVE MOBILE CALENDAR */
.mobile-calendar-responsive {
    width: 100%;
    overflow: hidden;
    border-radius: 8px;
    border: 1px solid var(--gray-200);
    background: white;
}

.calendar-grid {
    display: flex;
    flex-direction: column;
    width: 100%;
}

.calendar-header {
    display: grid;
    grid-template-columns: repeat(7, 1fr);
    background: var(--gray-50);
    border-bottom: 1px solid var(--gray-200);
}

.calendar-day-header {
    text-align: center;
    font-weight: 600;
    font-size: 0.75rem;
    color: var(--gray-600);
    padding: 8px 0;
    text-transform: uppercase;
}

.calendar-week {
    display: grid;
    grid-template-columns: repeat(7, 1fr);
    min-height: 70px;
}

.calendar-day-cell {
    border-right: 1px solid var(--gray-200);
    border-bottom: 1px solid var(--gray-200);
    position: relative;
    min-height: 70px;
}

.calendar-day-cell:nth-child(7n) {
    border-right: none;
}

.calendar-date {
    height: 100%;
    padding: 4px;
    position: relative;
}

.calendar-date.active {
    cursor: pointer;
    transition: background-color 0.2s;
}

.calendar-date.active:hover {
    background-color: var(--gray-50);
}

.calendar-date.empty {
    background-color: var(--gray-50);
}

.date-number {
    font-weight: 600;
    font-size: 0.85rem;
    color: var(--gray-800);
    margin-bottom: 2px;
    text-align: center;
}

/* MODIFICATION: Sunday (first column) color */
.calendar-day-cell:nth-child(1) .date-number {
    color: var(--danger) !important;
}

/* MODIFICATION: Saturday (last column) color */
.calendar-day-cell:nth-child(7n) .date-number {
    color: var(--info) !important;
}

/* MODIFICATION: Desktop Sunday color */
#calendar-body tr td:nth-child(1) .date-box {
    color: var(--danger) !important;
}

/* MODIFICATION: Desktop Saturday color */
#calendar-body tr td:nth-child(7) .date-box {
    color: var(--info) !important;
}

/* Event styling in mobile */
.calendar-date.active .event-placeholder {
    display: flex;
    flex-direction: column;
    gap: 2px;
    max-height: 50px;
    overflow-y: auto;
}

.calendar-date.active .event {
    background: var(--primary-orange);
    color: white;
    padding: 2px 4px;
    border-radius: 3px;
    font-size: 0.7rem;
    margin: 1px 0;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
    cursor: pointer;
}

/* Export container styling */
#schedule-container-mobile {
    background: white;
    padding: 20px;
    border-radius: 8px;
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
}

#schedule-container-mobile table {
    width: 100%;
    table-layout: fixed;
}

#schedule-container-mobile th,
#schedule-container-mobile td {
    vertical-align: top;
}

#schedule-container-mobile .event-placeholder-export {
    min-height: 70px;
}

#schedule-container-mobile .event {
    background: var(--primary-orange);
    color: white;
    padding: 0.15rem 0.4rem;
    border-radius: 4px;
    font-size: 0.75rem;
    margin: 0.1rem 0;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
}

/* Responsive adjustments */
@media (max-width: 576px) {
    .calendar-week {
        min-height: 65px;
    }
    
    .calendar-day-cell {
        min-height: 65px;
    }
    
    .date-number {
        font-size: 0.8rem;
    }
    
    .calendar-date.active .event {
        font-size: 0.65rem;
        padding: 1px 3px;
    }
}

@media (max-width: 400px) {
    .calendar-week {
        min-height: 60px;
    }
    
    .calendar-day-cell {
        min-height: 60px;
    }
    
    .date-number {
        font-size: 0.75rem;
    }
    
    .calendar-day-header {
        font-size: 0.7rem;
        padding: 6px 0;
    }
}

/* Scroll styling for events */
.calendar-date.active .event-placeholder::-webkit-scrollbar {
    width: 3px;
}

.calendar-date.active .event-placeholder::-webkit-scrollbar-track {
    background: transparent;
}

.calendar-date.active .event-placeholder::-webkit-scrollbar-thumb {
    background: var(--gray-300);
    border-radius: 3px;
}

/* Friday normal color */
.calendar-day-cell:nth-child(6n) .date-number {
    color: var(--gray-800);
}

/* Today highlight */
.calendar-date.active.today .date-number {
    background: var(--primary-orange);
    color: white;
    width: 24px;
    height: 24px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    margin: 0 auto 4px auto;
}

/* MODIFICATION: Selected color button styling */
.color-option.selected {
    transform: scale(1.05);
    border: 2px solid white !important;
    box-shadow: 0 0 0 2px var(--gray-700) !important;
    transition: all 0.2s ease;
}

.color-option {
    transition: all 0.2s ease;
    border: 2px solid transparent;
}

/* MODIFICATION: Color category layout */
.color-category-container {
    display: flex;
    flex-direction: column;
    gap: 8px;
}

.color-category-row {
    display: flex;
    flex-wrap: wrap;
    gap: 8px;
    justify-content: space-between;
}

.color-option {
    flex: 1;
    min-width: calc(25% - 6px);
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    padding: 8px 4px;
    border-radius: 6px;
    cursor: pointer;
    transition: all 0.2s ease;
}

.color-option:hover {
    transform: translateY(-2px);
}

.color-option .color-icon {
    font-size: 1.2rem;
    margin-bottom: 4px;
}

.color-option .color-label {
    font-size: 0.75rem;
    font-weight: 500;
}
</style>
@endsection

<!-- Modals (ENGLISH) -->
@push('modals')
<!-- Add Event Modal -->
<div class="modal fade" id="addEventModal" tabindex="-1" aria-labelledby="addEventModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="addEventModalLabel">Add Event</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div class="mb-3">
                    <label class="form-label">Date</label>
                    <input type="text" class="form-control" id="modal-date" readonly>
                </div>
                <div class="mb-3">
                    <label for="eventTitle" class="form-label">Event Title</label>
                    <input type="text" class="form-control" id="eventTitle" placeholder="e.g., Team Meeting">
                </div>
                <div class="mb-3">
                    <label class="form-label">Category / Color</label>
                    <div class="color-category-container">
                        <div class="color-category-row">
                            <button type="button" 
                                    class="btn btn-sm color-option selected" 
                                    style="background:#ef4444; color:white;"
                                    onclick="selectColor('urgent', this)">
                            </button>
                            <button type="button" 
                                    class="btn btn-sm color-option" 
                                    style="background:#f59e0b; color:white;"
                                    onclick="selectColor('default', this)">
                            </button>
                            <button type="button" 
                                    class="btn btn-sm color-option" 
                                    style="background:#10b981; color:white;"
                                    onclick="selectColor('done', this)">
                            </button>
                            <button type="button" 
                                    class="btn btn-sm color-option" 
                                    style="background:#3b82f6; color:white;"
                                    onclick="selectColor('important', this)">
                            </button>
                        </div>
                    </div>
                    <input type="hidden" id="selectedColor" value="urgent">
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" style="background: var(--accent-blue);" onclick="addEvent()">
                    <i class="fas fa-plus me-1"></i> Add Event
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Delete Event Modal -->
<div class="modal fade" id="deleteEventModal" tabindex="-1" aria-labelledby="deleteEventModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title text-danger" id="deleteEventModalLabel">Delete Event</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <p>Are you sure you want to delete this event?</p>
                <p class="fw-bold text-primary" id="deleteEventTitle">-</p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-danger" onclick="confirmDeleteEvent()">
                    <i class="fas fa-trash me-1"></i> Delete
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Delete All Modal -->
<div class="modal fade" id="deleteAllModal" tabindex="-1" aria-labelledby="deleteAllModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title text-danger" id="deleteAllModalLabel">Delete All Events</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <p class="text-muted">
                    <i class="fas fa-exclamation-triangle text-warning me-1"></i>
                    This action will delete <strong>all events this month</strong>. Cannot be undone.
                </p>
                <div class="form-check mt-3">
                    <input class="form-check-input" type="checkbox" id="confirmDeleteAllCheck">
                    <label class="form-check-label" for="confirmDeleteAllCheck">
                        I understand and want to proceed
                    </label>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-danger" id="confirmDeleteAllBtn" disabled onclick="confirmDeleteAll()">
                    <i class="fas fa-trash me-1"></i> Delete All
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Upload to Server Modal -->
<div class="modal fade" id="uploadModal" tabindex="-1" aria-labelledby="uploadModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="uploadModalLabel">
                    <i class="fas fa-cloud-upload-alt me-2"></i> 
                    <span id="modalUploadTitle">Save Schedule to Server</span>
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div class="alert alert-info" id="uploadInfoAlert">
                    <i class="fas fa-info-circle me-2"></i>
                    <span id="uploadInfoText">Schedule will be saved as an image to the server.</span>
                </div>
                <input type="hidden" id="currentScheduleId">
                <input type="hidden" id="isUpdateMode" value="false">
                <form id="uploadForm">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label fw-medium">Schedule Title</label>
                                <input type="text" class="form-control" id="uploadTitleInput" readonly>
                            </div>
                            <div class="mb-3">
                                <label class="form-label fw-medium">Month & Year</label>
                                <input type="text" class="form-control" id="uploadMonthYear" readonly>
                            </div>
                            <div class="mb-3">
                                <label class="form-label fw-medium">Schedule Info</label>
                                <div class="border rounded p-3">
                                    <p class="mb-1">
                                        <i class="fas fa-calendar-alt me-2"></i>
                                        <strong>Month:</strong> <span id="uploadMonth"></span>
                                    </p>
                                    <p class="mb-1">
                                        <i class="fas fa-calendar-check me-2"></i>
                                        <strong>Year:</strong> <span id="uploadYear"></span>
                                    </p>
                                    <p class="mb-0">
                                        <i class="fas fa-tasks me-2"></i>
                                        <strong>Total Events:</strong> <span id="uploadTotalEvents">0</span>
                                    </p>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label fw-medium">Image Preview</label>
                                <div class="text-center border rounded p-3 bg-light">
                                    <div id="uploadPreviewContainer">
                                        <img id="uploadPreview" src="" alt="Preview"
                                             class="img-fluid rounded border mb-2" style="max-height: 180px;">
                                        <small class="text-muted d-block">Format: PNG | Quality: High</small>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                    <i class="fas fa-times me-1"></i> Cancel
                </button>
                <button type="button" class="btn btn-primary" id="uploadToServerBtn" onclick="uploadToServer()">
                    <i class="fas fa-cloud-upload-alt me-1"></i> Save to Server
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Delete Schedule Confirmation Modal -->
<div class="modal fade" id="deleteScheduleModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header bg-danger text-white">
                <h5 class="modal-title">
                    <i class="fas fa-trash-alt me-2"></i> Delete Schedule
                </h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <p>Are you sure you want to delete this schedule?</p>
                <p class="fw-bold" id="deleteScheduleTitle">-</p>
                <p class="text-muted small">Data will be deleted from the server and cannot be restored.</p>
                <input type="hidden" id="deleteScheduleId">
                <input type="hidden" id="deleteScheduleIsServer">
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-danger" onclick="confirmDeleteSchedule()">
                    <i class="fas fa-trash me-1"></i> Delete
                </button>
            </div>
        </div>
    </div>
</div>
@endpush

@section('scripts')
<script src="https://cdnjs.cloudflare.com/ajax/libs/html2canvas/1.4.1/html2canvas.min.js"></script>
<script>
(function() {
    'use strict';

    // ✅ GLOBAL STATE WITH UPDATE SUPPORT
    window.calendarState = {
        dateKey: null,
        deleteTarget: null,
        events: JSON.parse(localStorage.getItem('calendarEvents_global')) || {},
        monthNames: ["January","February","March","April","May","June","July","August","September","October","November","December"],
        dayNames: ["Sunday","Monday","Tuesday","Wednesday","Thursday","Friday","Saturday"],
        colors: { 
            'urgent': '#ef4444',
            'default': '#f59e0b',
            'done': '#10b981',
            'important': '#3b82f6'
        },
        colorLabels: {
            'urgent': 'Urgent',
            'default': 'Normal', 
            'done': 'Done',
            'important': 'Important'
        },
        current: new Date({{ $year }}, {{ $month - 1 }}, 1),
        currentScheduleId: null,
        isEditingExisting: false
    };

    // ✅ API Configuration with update endpoint
    const API_CONFIG = {
        baseUrl: '{{ url("/") }}',
        endpoints: {
            store: '{{ route("jadwal.store") }}',
            index: '{{ route("jadwal.index") }}',
            show: (id) => `{{ route("jadwal.show", "") }}/${id}`,
            destroy: (id) => `{{ route("jadwal.destroy", "") }}/${id}`,
            update: (id) => `{{ route("jadwal.update", "") }}/${id}`,
        },
        csrfToken: document.querySelector('meta[name="csrf-token"]').getAttribute('content')
    };

    // ✅ Helper Functions
    function getDaysInMonth(y, m) {
        const monthLengths = [31, (y % 4 === 0 && y % 100 !== 0) || y % 400 === 0 ? 29 : 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];
        return monthLengths[m];
    }

    function getFirstDay(y, m) {
        return new Date(y, m, 1).getDay();
    }

    function showAlert(message, type = 'success', duration = 3000) {
        Swal.fire({
            toast: true,
            position: 'top-end',
            icon: type,
            title: message,
            showConfirmButton: false,
            timer: duration,
            timerProgressBar: true,
            background: 'var(--white)',
            color: 'var(--gray-800)',
            iconColor: type === 'success' ? 'var(--success)' :
                      type === 'error' ? 'var(--danger)' :
                      type === 'warning' ? 'var(--warning)' : 'var(--info)'
        });
    }

    // ✅ Color selection
    window.selectColor = (color, buttonElement) => {
        document.getElementById('selectedColor').value = color;
        document.querySelectorAll('.color-option').forEach(btn => btn.classList.remove('selected'));
        buttonElement.classList.add('selected');
    };

    // ✅ Render Events
    function renderEvents() {
        document.querySelectorAll('.event-placeholder').forEach(el => {
            const date = parseInt(el.dataset.date);
            const y = window.calendarState.current.getFullYear();
            const m = window.calendarState.current.getMonth();
            const key = `${y}-${String(m+1).padStart(2,'0')}-${String(date).padStart(2,'0')}`;

            el.innerHTML = '';
            const events = window.calendarState.events[key] || [];

            events.forEach((ev, idx) => {
                const div = document.createElement('div');
                div.className = 'event';
                div.style.cssText = `
                    background: ${window.calendarState.colors[ev.color || 'default']};
                    color: white;
                    padding: 0.15rem 0.4rem;
                    border-radius: 4px;
                    font-size: 0.75rem;
                    margin: 0.1rem 0;
                    cursor: pointer;
                    white-space: nowrap;
                    overflow: hidden;
                    text-overflow: ellipsis;
                    transition: transform 0.2s;
                `;
                div.title = ev.title;
                div.textContent = ev.title;

                div.onclick = (e) => {
                    e.stopPropagation();
                    document.getElementById('deleteEventTitle').textContent = ev.title;
                    window.calendarState.deleteTarget = { key, idx };
                    new bootstrap.Modal(document.getElementById('deleteEventModal')).show();
                };

                el.appendChild(div);
            });

            const cell = el.closest('.calendar-date.active') || el.parentElement;
            cell.style.cursor = 'pointer';
            cell.onclick = (e) => {
                if (e.target.closest('.event')) return;
                const monthName = window.calendarState.monthNames[m];
                document.getElementById('modal-date').value = `${date} ${monthName} ${y}`;
                window.calendarState.dateKey = key;

                document.getElementById('eventTitle').value = '';
                document.getElementById('selectedColor').value = 'urgent';
                
                document.querySelectorAll('.color-option').forEach(btn => {
                    btn.classList.remove('selected');
                });
                
                const firstColorBtn = document.querySelector('.color-option');
                if (firstColorBtn) {
                    firstColorBtn.classList.add('selected');
                }

                new bootstrap.Modal(document.getElementById('addEventModal')).show();
            };
        });

        const today = new Date();
        if (window.calendarState.current.getMonth() === today.getMonth() && 
            window.calendarState.current.getFullYear() === today.getFullYear()) {
            
            document.querySelectorAll('.calendar-date.active').forEach(dateCell => {
                const dateNumber = dateCell.querySelector('.date-number');
                if (dateNumber && parseInt(dateNumber.textContent) === today.getDate()) {
                    dateCell.classList.add('today');
                }
            });
        }
    }

    // ✅ Render Desktop Calendar
    function renderDesktopCalendar() {
        const y = window.calendarState.current.getFullYear();
        const m = window.calendarState.current.getMonth();
        const daysInMonth = getDaysInMonth(y, m);
        const firstDay = getFirstDay(y, m);
        
        let html = '';
        let date = 1;
        
        for (let week = 0; week < 6; week++) {
            if (date > daysInMonth) break;
            
            html += '<tr>';
            
            for (let dayIndex = 0; dayIndex < 7; dayIndex++) {
                html += '<td style="padding:0.4rem; border:1px solid var(--gray-200); text-align:left; height:95px;">';
                
                if (week === 0 && dayIndex < firstDay) {
                    // empty
                } else if (date <= daysInMonth) {
                    const key = `${y}-${String(m+1).padStart(2,'0')}-${String(date).padStart(2,'0')}`;
                    const events = window.calendarState.events[key] || [];
                    let eventsHtml = '';
                    
                    events.forEach((ev, idx) => {
                        eventsHtml += `
                            <div class="event" 
                                 style="background:${window.calendarState.colors[ev.color || 'default']}; color:white; padding:0.15rem 0.4rem; border-radius:4px; font-size:0.75rem; margin:0.1rem 0; cursor:pointer;"
                                 onclick="event.stopPropagation(); window.calendarState.deleteTarget = {key: '${key}', idx: ${idx}}; document.getElementById('deleteEventTitle').textContent = '${ev.title.replace(/'/g, "\\'")}'; new bootstrap.Modal(document.getElementById('deleteEventModal')).show();">
                                ${ev.title}
                            </div>
                        `;
                    });
                    
                    html += `
                        <div class="date-box" style="font-weight:600; margin-bottom:0.25rem; ${dayIndex === 0 ? 'color: var(--danger)!important;' : dayIndex === 6 ? 'color: var(--info)!important;' : ''}">
                            ${date}
                        </div>
                        <div class="event-placeholder" data-date="${date}" onclick="
                            const monthName = window.calendarState.monthNames[${m}];
                            document.getElementById('modal-date').value = '${date} ' + monthName + ' ${y}';
                            window.calendarState.dateKey = '${key}';
                            document.getElementById('eventTitle').value = '';
                            document.getElementById('selectedColor').value = 'urgent';
                            
                            document.querySelectorAll('.color-option').forEach(btn => {
                                btn.classList.remove('selected');
                            });
                            
                            const firstColorBtn = document.querySelector('.color-option');
                            if (firstColorBtn) {
                                firstColorBtn.classList.add('selected');
                            }
                            
                            new bootstrap.Modal(document.getElementById('addEventModal')).show();
                        " style="cursor:pointer;">
                            ${eventsHtml}
                        </div>
                    `;
                    date++;
                }
                
                html += '</td>';
            }
            
            html += '</tr>';
        }
        
        const calendarBody = document.getElementById('calendar-body');
        if (calendarBody) {
            calendarBody.innerHTML = html;
        }
        
        updateMobileExportContainer();
    }

    // ✅ Render Header
    function renderHeader() {
        const y = window.calendarState.current.getFullYear();
        const m = window.calendarState.current.getMonth();
        const text = `${window.calendarState.monthNames[m]} ${y}`;
        const title = `MOD Schedule ${text}`;

        ['month-year', 'month-year-mobile'].forEach(id => {
            const el = document.getElementById(id);
            if (el) el.textContent = text;
        });

        ['calendar-title', 'calendar-title-mobile'].forEach(id => {
            const el = document.getElementById(id);
            if (el) el.value = title;
        });
        ['calendar-title-display', 'calendar-title-display-mobile'].forEach(id => {
            const el = document.getElementById(id);
            if (el) el.textContent = title;
        });
    }

    // ✅ Render Mobile Calendar
    function renderMobileCalendar() {
        const container = document.querySelector('.calendar-grid');
        if (!container) return;

        const y = window.calendarState.current.getFullYear();
        const m = window.calendarState.current.getMonth();
        const daysInMonth = getDaysInMonth(y, m);
        const firstDay = getFirstDay(y, m);
        
        let html = `
            <div class="calendar-header">
                <div class="calendar-day-header">S</div>
                <div class="calendar-day-header">M</div>
                <div class="calendar-day-header">T</div>
                <div class="calendar-day-header">W</div>
                <div class="calendar-day-header">T</div>
                <div class="calendar-day-header">F</div>
                <div class="calendar-day-header">S</div>
            </div>
        `;

        let date = 1;
        for (let week = 0; week < 6; week++) {
            if (date > daysInMonth) break;
            
            html += '<div class="calendar-week">';
            
            for (let dayIndex = 0; dayIndex < 7; dayIndex++) {
                html += '<div class="calendar-day-cell">';
                
                if (week === 0 && dayIndex < firstDay) {
                    html += '<div class="calendar-date empty"></div>';
                } else if (date <= daysInMonth) {
                    const key = `${y}-${String(m+1).padStart(2,'0')}-${String(date).padStart(2,'0')}`;
                    const events = window.calendarState.events[key] || [];
                    let eventsHtml = '';
                    
                    events.forEach((ev, idx) => {
                        eventsHtml += `
                            <div class="event" 
                                 style="background:${window.calendarState.colors[ev.color || 'default']}"
                                 onclick="event.stopPropagation(); window.calendarState.deleteTarget = {key: '${key}', idx: ${idx}}; document.getElementById('deleteEventTitle').textContent = '${ev.title.replace(/'/g, "\\'")}'; new bootstrap.Modal(document.getElementById('deleteEventModal')).show();">
                                ${ev.title}
                            </div>
                        `;
                    });
                    
                    html += `
                        <div class="calendar-date active" onclick="
                            const monthName = window.calendarState.monthNames[${m}];
                            document.getElementById('modal-date').value = '${date} ' + monthName + ' ${y}';
                            window.calendarState.dateKey = '${key}';
                            document.getElementById('eventTitle').value = '';
                            document.getElementById('selectedColor').value = 'urgent';
                            
                            document.querySelectorAll('.color-option').forEach(btn => {
                                btn.classList.remove('selected');
                            });
                            
                            const firstColorBtn = document.querySelector('.color-option');
                            if (firstColorBtn) {
                                firstColorBtn.classList.add('selected');
                            }
                            
                            new bootstrap.Modal(document.getElementById('addEventModal')).show();
                        ">
                            <div class="date-number">${date}</div>
                            <div class="event-placeholder" data-date="${date}">
                                ${eventsHtml}
                            </div>
                        </div>
                    `;
                    date++;
                } else {
                    html += '<div class="calendar-date empty"></div>';
                }
                
                html += '</div>';
            }
            
            html += '</div>';
        }
        
        container.innerHTML = html;
        
        const today = new Date();
        if (m === today.getMonth() && y === today.getFullYear()) {
            setTimeout(() => {
                document.querySelectorAll('.calendar-date.active').forEach(dateCell => {
                    const dateNumber = dateCell.querySelector('.date-number');
                    if (dateNumber && parseInt(dateNumber.textContent) === today.getDate()) {
                        dateCell.classList.add('today');
                    }
                });
            }, 100);
        }
    }

    // ✅ Render All
    window.renderAll = function() {
        renderHeader();
        
        if (window.innerWidth < 992) {
            renderMobileCalendar();
        } else {
            renderDesktopCalendar();
            renderEvents();
        }
    };

    // ✅ Global Functions
    window.changeMonth = (delta) => {
        const d = window.calendarState.current;
        window.calendarState.current = new Date(d.getFullYear(), d.getMonth() + delta, 1);
        window.renderAll();
    };

    window.addEvent = () => {
        const title = document.getElementById('eventTitle').value.trim();
        const color = document.getElementById('selectedColor').value;
        const key = window.calendarState.dateKey;

        if (!title) {
            showAlert('Event title cannot be empty', 'warning');
            return;
        }

        if (!window.calendarState.events[key]) {
            window.calendarState.events[key] = [];
        }
        window.calendarState.events[key].push({ title, color });
        localStorage.setItem('calendarEvents_global', JSON.stringify(window.calendarState.events));
        window.renderAll();

        bootstrap.Modal.getInstance(document.getElementById('addEventModal')).hide();
        showAlert('Event added successfully');
    };

    window.confirmDeleteEvent = () => {
        const { key, idx } = window.calendarState.deleteTarget || {};
        if (key && window.calendarState.events[key]) {
            window.calendarState.events[key].splice(idx, 1);
            if (window.calendarState.events[key].length === 0) {
                delete window.calendarState.events[key];
            }
            localStorage.setItem('calendarEvents_global', JSON.stringify(window.calendarState.events));
            window.renderAll();
            showAlert('Event deleted successfully');
        }
        bootstrap.Modal.getInstance(document.getElementById('deleteEventModal')).hide();
    };

    window.confirmDeleteAll = () => {
        const y = window.calendarState.current.getFullYear();
        const m = window.calendarState.current.getMonth();
        const days = getDaysInMonth(y, m);
        let totalDeleted = 0;

        for (let d = 1; d <= days; d++) {
            const key = `${y}-${String(m+1).padStart(2,'0')}-${String(d).padStart(2,'0')}`;
            if (window.calendarState.events[key]) {
                totalDeleted += window.calendarState.events[key].length;
                delete window.calendarState.events[key];
            }
        }

        localStorage.setItem('calendarEvents_global', JSON.stringify(window.calendarState.events));
        window.renderAll();
        bootstrap.Modal.getInstance(document.getElementById('deleteAllModal')).hide();
        showAlert(`${totalDeleted} events deleted successfully`);
    };

    // ✅ Update mobile export container
    function updateMobileExportContainer() {
        const y = window.calendarState.current.getFullYear();
        const m = window.calendarState.current.getMonth();
        const daysInMonth = getDaysInMonth(y, m);
        const firstDay = getFirstDay(y, m);
        
        const title = window.innerWidth >= 992 
            ? document.getElementById('calendar-title').value
            : document.getElementById('calendar-title-mobile').value;
        
        const titleDisplay = document.getElementById('calendar-title-display-mobile');
        if (titleDisplay) {
            titleDisplay.textContent = title;
        }
        
        document.querySelectorAll('#calendar-body-hidden .event-placeholder-export').forEach(el => {
            const date = parseInt(el.dataset.date);
            const key = `${y}-${String(m+1).padStart(2,'0')}-${String(date).padStart(2,'0')}`;
            const events = window.calendarState.events[key] || [];
            
            el.innerHTML = '';
            events.forEach((ev, idx) => {
                const div = document.createElement('div');
                div.className = 'event';
                div.style.cssText = `
                    background: ${window.calendarState.colors[ev.color || 'default']};
                    color: white;
                    padding: 0.15rem 0.4rem;
                    border-radius: 4px;
                    font-size: 0.75rem;
                    margin: 0.1rem 0;
                    white-space: nowrap;
                    overflow: hidden;
                    text-overflow: ellipsis;
                `;
                div.textContent = ev.title;
                el.appendChild(div);
            });
        });
    }

    // ✅ Save Locally (Download)
    window.simpanLokal = () => {
        let container;
        
        if (window.innerWidth >= 992) {
            container = document.getElementById('schedule-container');
        } else {
            container = document.getElementById('schedule-container-mobile');
            updateMobileExportContainer();
        }
        
        if (!container) return showAlert('Container not found', 'error');

        const originalDisplay = container.style.display;
        const originalWidth = container.style.width;
        
        if (window.innerWidth < 992) {
            container.style.display = 'block';
            container.style.width = '800px';
        }

        html2canvas(container, {
            scale: 2,
            backgroundColor: '#fff',
            logging: false,
            width: container.scrollWidth,
            height: container.scrollHeight,
            useCORS: true,
            allowTaint: true
        }).then(canvas => {
            const a = document.createElement('a');
            const month = window.calendarState.monthNames[window.calendarState.current.getMonth()];
            const year = window.calendarState.current.getFullYear();
            const title = window.innerWidth >= 992 
                ? document.getElementById('calendar-title').value
                : document.getElementById('calendar-title-mobile').value;
            const filename = `${title.replace(/\s+/g, '_')}_${month}_${year}`;
            a.download = `${filename.toLowerCase()}.png`;
            a.href = canvas.toDataURL('image/png');
            a.click();
            showAlert('Schedule downloaded successfully');
            
            if (window.innerWidth < 992) {
                container.style.display = originalDisplay;
                container.style.width = originalWidth;
            }
        }).catch(error => {
            console.error('Error generating image:', error);
            showAlert('Failed to download schedule', 'error');
            
            if (window.innerWidth < 992) {
                container.style.display = originalDisplay;
                container.style.width = originalWidth;
            }
        });
    };

    // ✅ Prepare Upload
    window.prepareUpload = () => {
        let container;
        
        if (window.innerWidth >= 992) {
            container = document.getElementById('schedule-container');
        } else {
            container = document.getElementById('schedule-container-mobile');
            updateMobileExportContainer();
        }
        
        if (!container) return showAlert('Container not found', 'error');

        const originalDisplay = container.style.display;
        const originalWidth = container.style.width;
        
        if (window.innerWidth < 992) {
            container.style.display = 'block';
            container.style.width = '800px';
        }

        const previewContainer = document.getElementById('uploadPreviewContainer');
        previewContainer.innerHTML = `
            <div class="text-center py-4">
                <div class="spinner-border text-primary mb-3" role="status">
                    <span class="visually-hidden">Loading...</span>
                </div>
                <p class="text-muted">Generating preview...</p>
            </div>
        `;

        html2canvas(container, {
            scale: 2,
            backgroundColor: '#fff',
            logging: false,
            width: container.scrollWidth,
            height: container.scrollHeight,
            useCORS: true,
            allowTaint: true
        }).then(canvas => {
            const dataUrl = canvas.toDataURL('image/png');
            const title = window.innerWidth >= 992 
                ? document.getElementById('calendar-title').value
                : document.getElementById('calendar-title-mobile').value;
            const month = window.calendarState.monthNames[window.calendarState.current.getMonth()];
            const year = window.calendarState.current.getFullYear();

            const modTitle = title.trim() || `MOD Schedule ${month} ${year}`;

            let totalEvents = 0;
            Object.values(window.calendarState.events).forEach(events => {
                totalEvents += events.length;
            });

            document.getElementById('uploadTitleInput').value = modTitle;
            document.getElementById('uploadMonthYear').value = `${month} ${year}`;
            document.getElementById('uploadMonth').textContent = month;
            document.getElementById('uploadYear').textContent = year;
            document.getElementById('uploadTotalEvents').textContent = totalEvents;

            previewContainer.innerHTML = `
                <img id="uploadPreview" src="${dataUrl}" alt="Preview"
                     class="img-fluid rounded border mb-2" style="max-height: 180px;">
                <small class="text-muted d-block">Format: PNG | Quality: High</small>
            `;

            const modal = document.getElementById('uploadModal');
            modal.dataset.title = modTitle;
            modal.dataset.month = month;
            modal.dataset.year = year;
            modal.dataset.image = dataUrl;
            
            if (window.calendarState.currentScheduleId && window.calendarState.isEditingExisting) {
                document.getElementById('modalUploadTitle').textContent = 'Update Schedule on Server';
                document.getElementById('uploadInfoText').textContent = 'Existing schedule will be updated with new data.';
                document.getElementById('uploadInfoAlert').className = 'alert alert-warning';
                document.getElementById('uploadToServerBtn').innerHTML = '<i class="fas fa-sync me-1"></i> Update Schedule';
                document.getElementById('currentScheduleId').value = window.calendarState.currentScheduleId;
                document.getElementById('isUpdateMode').value = 'true';
            } else {
                document.getElementById('modalUploadTitle').textContent = 'Save Schedule to Server';
                document.getElementById('uploadInfoText').textContent = 'Schedule will be saved as an image to the server.';
                document.getElementById('uploadInfoAlert').className = 'alert alert-info';
                document.getElementById('uploadToServerBtn').innerHTML = '<i class="fas fa-cloud-upload-alt me-1"></i> Save to Server';
                document.getElementById('currentScheduleId').value = '';
                document.getElementById('isUpdateMode').value = 'false';
            }

            if (window.innerWidth < 992) {
                container.style.display = originalDisplay;
                container.style.width = originalWidth;
            }

        }).catch(error => {
            console.error('Error generating preview:', error);
            previewContainer.innerHTML = `
                <div class="alert alert-danger">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    Failed to generate preview
                </div>
            `;
            
            if (window.innerWidth < 992) {
                container.style.display = originalDisplay;
                container.style.width = originalWidth;
            }
        });
    };

    // ✅ Upload to Server
    window.uploadToServer = async () => {
        const modal = document.getElementById('uploadModal');
        const title = modal.dataset.title || 'Schedule';
        const month = modal.dataset.month;
        const year = modal.dataset.year;
        const imageBase64 = modal.dataset.image;
        const isUpdateMode = document.getElementById('isUpdateMode').value === 'true';
        const scheduleId = document.getElementById('currentScheduleId').value;

        if (!imageBase64) {
            showAlert('Image not found', 'error');
            return;
        }

        const data = {
            title: title,
            month: month,
            year: parseInt(year),
            events: JSON.stringify(window.calendarState.events),
            image: imageBase64
        };

        if (isUpdateMode && scheduleId) {
            data._method = 'PUT';
        }

        const uploadBtn = document.getElementById('uploadToServerBtn');
        const originalText = uploadBtn.innerHTML;
        uploadBtn.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span> Saving...';
        uploadBtn.disabled = true;

        try {
            let url, method;
            
            if (isUpdateMode && scheduleId) {
                url = API_CONFIG.endpoints.update(scheduleId);
                method = 'POST';
            } else {
                url = API_CONFIG.endpoints.store;
                method = 'POST';
            }

            const response = await fetch(url, {
                method: method,
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': API_CONFIG.csrfToken,
                    'Accept': 'application/json'
                },
                body: JSON.stringify(data)
            });

            const result = await response.json();

            if (result.success) {
                const jadwal = {
                    id: result.data.id,
                    title: result.data.title,
                    month: month,
                    year: year,
                    events: window.calendarState.events,
                    image_url: result.data.image_url,
                    created_at: result.data.created_at,
                    is_from_server: true
                };

                let jadwals = JSON.parse(localStorage.getItem('jadwal_global')) || [];
                
                if (isUpdateMode && scheduleId) {
                    const index = jadwals.findIndex(j => j.id == scheduleId);
                    if (index !== -1) {
                        jadwals[index] = jadwal;
                    } else {
                        jadwals.push(jadwal);
                    }
                } else {
                    jadwals.push(jadwal);
                }
                
                localStorage.setItem('jadwal_global', JSON.stringify(jadwals));

                window.calendarState.currentScheduleId = null;
                window.calendarState.isEditingExisting = false;

                bootstrap.Modal.getInstance(document.getElementById('uploadModal')).hide();
                showAlert(`Schedule ${isUpdateMode ? 'updated' : 'saved'} successfully!`);

                window.muatGaleri();
            } else {
                if (result.message && result.message.includes('already exists') && !isUpdateMode) {
                    Swal.fire({
                        title: 'Schedule Already Exists',
                        text: result.message + ' Do you want to overwrite it?',
                        icon: 'warning',
                        showCancelButton: true,
                        confirmButtonText: 'Yes, Overwrite',
                        cancelButtonText: 'Cancel',
                        reverseButtons: true
                    }).then((result) => {
                        if (result.isConfirmed) {
                            let jadwals = JSON.parse(localStorage.getItem('jadwal_global')) || [];
                            const existingSchedule = jadwals.find(j => 
                                j.month === month && j.year === year && j.is_from_server
                            );
                            
                            if (existingSchedule) {
                                window.calendarState.currentScheduleId = existingSchedule.id;
                                window.calendarState.isEditingExisting = true;
                                document.getElementById('currentScheduleId').value = existingSchedule.id;
                                document.getElementById('isUpdateMode').value = 'true';
                                
                                window.uploadToServer();
                            }
                        }
                    });
                } else {
                    throw new Error(result.message || 'Failed to save to server');
                }
            }
        } catch (error) {
            console.error('Upload error:', error);
            Swal.fire({
                title: 'Save Failed',
                text: error.message,
                icon: 'error',
                confirmButtonText: 'OK'
            });
        } finally {
            uploadBtn.innerHTML = originalText;
            uploadBtn.disabled = false;
        }
    };

    // ✅ Load Gallery
    window.muatGaleri = async () => {
        const containerId = window.innerWidth >= 992 ? 'gallery-container' : 'gallery-container-mobile';
        const container = document.getElementById(containerId);
        if (!container) return;

        container.innerHTML = `
            <div class="col-12 text-center py-4">
                <div class="spinner-border text-primary mb-3" role="status">
                    <span class="visually-hidden">Loading...</span>
                </div>
                <p class="text-muted">Loading schedules...</p>
            </div>
        `;

        try {
            const serverResponse = await fetch(API_CONFIG.endpoints.index);
            const serverResult = await serverResponse.json();
            const serverSchedules = serverResult.success ? serverResult.data : [];

            const localSchedules = JSON.parse(localStorage.getItem('jadwal_global')) || [];

            const allSchedules = [...serverSchedules];
            localSchedules.forEach(local => {
                if (!allSchedules.some(s => s.id == local.id)) {
                    allSchedules.push(local);
                }
            });

            if (allSchedules.length === 0) {
                container.innerHTML = `
                    <div class="col-12 text-center py-4">
                        <i class="fas fa-calendar-alt fa-3x text-muted mb-3"></i>
                        <p class="text-muted">No schedules saved yet</p>
                    </div>
                `;
                return;
            }

            allSchedules.sort((a, b) => new Date(b.created_at) - new Date(a.created_at));

            let html = '';
            allSchedules.forEach(schedule => {
                const isFromServer = schedule.is_from_server || (schedule.id && schedule.id > 1000);
                const imageUrl = schedule.image_url ||
                    `data:image/svg+xml;utf8,<svg xmlns='http://www.w3.org/2000/svg' width='200' height='120' viewBox='0 0 200 120'>
                        <rect width='200' height='120' fill='%23f1f5f9'/>
                        <rect width='200' height='30' fill='%23${isFromServer ? '10b981' : '6b7280'}'/>
                        <text x='10' y='55' font-size='12' fill='%231e293b' font-family='sans-serif'>${schedule.title}</text>
                        <text x='10' y='75' font-size='10' fill='%2364748b'>${schedule.month} ${schedule.year}</text>
                    </svg>`;

                html += `
                <div class="col-6 col-md-4 col-lg-3">
                    <div class="card border-0 shadow-sm h-100">
                        <div class="position-relative">
                            <img src="${imageUrl}" class="card-img-top" style="height:120px;object-fit:cover;">
                            <span class="badge ${isFromServer ? 'bg-success' : 'bg-secondary'} position-absolute top-0 start-0 m-2">
                                ${isFromServer ? '<i class="fas fa-server me-1"></i>Server' : '<i class="fas fa-laptop me-1"></i>Local'}
                            </span>
                            <button type="button" class="btn btn-danger btn-sm position-absolute top-0 end-0 m-2 rounded-circle"
                                    style="width:28px;height:28px;padding:0;"
                                    onclick="window.showDeleteScheduleModal('${schedule.id}', '${schedule.title}', ${isFromServer})">
                                <i class="fas fa-times" style="font-size:0.7rem;"></i>
                            </button>
                        </div>
                        <div class="card-body p-2">
                            <p class="card-text small fw-bold text-truncate">${schedule.title}</p>
                            <div class="d-flex justify-content-between align-items-center">
                                <small class="text-muted">${schedule.month} ${schedule.year}</small>
                                <button class="btn btn-sm btn-outline-info" onclick="window.muatJadwal('${schedule.id}', ${isFromServer})">
                                    <i class="fas fa-edit"></i>
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
                `;
            });

            container.innerHTML = html;

        } catch (error) {
            console.error('Error loading gallery:', error);
            container.innerHTML = `
                <div class="col-12">
                    <div class="alert alert-danger">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        Failed to load schedules from server
                        <button class="btn btn-sm btn-outline-primary mt-2" onclick="window.muatGaleriLokal()">
                            <i class="fas fa-laptop me-1"></i> Load from browser
                        </button>
                    </div>
                </div>
            `;
        }
    };

    // ✅ Load from LocalStorage only
    window.muatGaleriLokal = () => {
        const containerId = window.innerWidth >= 992 ? 'gallery-container' : 'gallery-container-mobile';
        const container = document.getElementById(containerId);

        const localSchedules = JSON.parse(localStorage.getItem('jadwal_global')) || [];

        if (localSchedules.length === 0) {
            container.innerHTML = `
                <div class="col-12 text-center py-4">
                    <i class="fas fa-calendar-alt fa-3x text-muted mb-3"></i>
                    <p class="text-muted">No schedules in browser</p>
                </div>
            `;
            return;
        }

        let html = '';
        localSchedules.forEach(schedule => {
            const imageUrl = `data:image/svg+xml;utf8,<svg xmlns='http://www.w3.org/2000/svg' width='200' height='120' viewBox='0 0 200 120'>
                <rect width='200' height='120' fill='%23f1f5f9'/>
                <rect width='200' height='30' fill='%236b7280'/>
                <text x='10' y='55' font-size='12' fill='%231e293b' font-family='sans-serif'>${schedule.title}</text>
                <text x='10' y='75' font-size='10' fill='%2364748b'>${schedule.month} ${schedule.year}</text>
            </svg>`;

            html += `
            <div class="col-6 col-md-4 col-lg-3">
                <div class="card border-0 shadow-sm h-100">
                    <div class="position-relative">
                        <img src="${imageUrl}" class="card-img-top" style="height:120px;object-fit:cover;">
                        <span class="badge bg-secondary position-absolute top-0 start-0 m-2">
                            <i class="fas fa-laptop me-1"></i>Local
                        </span>
                        <button type="button" class="btn btn-danger btn-sm position-absolute top-0 end-0 m-2 rounded-circle"
                                style="width:28px;height:28px;padding:0;"
                                onclick="window.showDeleteScheduleModal('${schedule.id}', '${schedule.title}', false)">
                            <i class="fas fa-times" style="font-size:0.7rem;"></i>
                        </button>
                    </div>
                    <div class="card-body p-2">
                        <p class="card-text small fw-bold text-truncate">${schedule.title}</p>
                        <div class="d-flex justify-content-between align-items-center">
                            <small class="text-muted">${schedule.month} ${schedule.year}</small>
                            <button class="btn btn-sm btn-outline-info" onclick="window.muatJadwal('${schedule.id}', false)">
                                <i class="fas fa-edit"></i>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
            `;
        });

        container.innerHTML = html;
    };

    // ✅ Show Delete Schedule Modal
    window.showDeleteScheduleModal = (id, title, isFromServer) => {
        document.getElementById('deleteScheduleId').value = id;
        document.getElementById('deleteScheduleTitle').textContent = title;
        document.getElementById('deleteScheduleIsServer').value = isFromServer;

        new bootstrap.Modal(document.getElementById('deleteScheduleModal')).show();
    };

    // ✅ Confirm Delete Schedule
    window.confirmDeleteSchedule = async () => {
        const id = document.getElementById('deleteScheduleId').value;
        const isFromServer = document.getElementById('deleteScheduleIsServer').value === 'true';
        const title = document.getElementById('deleteScheduleTitle').textContent;

        try {
            if (isFromServer) {
                const response = await fetch(API_CONFIG.endpoints.destroy(id), {
                    method: 'DELETE',
                    headers: {
                        'X-CSRF-TOKEN': API_CONFIG.csrfToken,
                        'Accept': 'application/json'
                    }
                });

                const result = await response.json();
                if (!result.success) {
                    throw new Error(result.message);
                }
            }

            let jadwals = JSON.parse(localStorage.getItem('jadwal_global')) || [];
            jadwals = jadwals.filter(j => j.id != id);
            localStorage.setItem('jadwal_global', JSON.stringify(jadwals));

            if (window.calendarState.currentScheduleId == id) {
                window.calendarState.currentScheduleId = null;
                window.calendarState.isEditingExisting = false;
            }

            bootstrap.Modal.getInstance(document.getElementById('deleteScheduleModal')).hide();

            showAlert('Schedule deleted successfully');

            window.muatGaleri();

        } catch (error) {
            console.error('Error deleting schedule:', error);
            showAlert('Failed to delete schedule: ' + error.message, 'error');
        }
    };

    // ✅ Load Schedule to Editor
    window.muatJadwal = async (id, isFromServer = false) => {
        let schedule;

        if (isFromServer) {
            try {
                const response = await fetch(API_CONFIG.endpoints.show(id));
                const result = await response.json();

                if (result.success) {
                    schedule = result.data;
                } else {
                    throw new Error(result.message);
                }
            } catch (error) {
                showAlert('Failed to load schedule from server: ' + error.message, 'error');
                return;
            }
        } else {
            const jadwals = JSON.parse(localStorage.getItem('jadwal_global')) || [];
            schedule = jadwals.find(j => j.id == id);
        }

        if (!schedule) {
            showAlert('Schedule not found', 'error');
            return;
        }

        const displayTitle = schedule.title || `Schedule ${schedule.month} ${schedule.year}`;

        const desktopTitle = document.getElementById('calendar-title');
        const mobileTitle = document.getElementById('calendar-title-mobile');
        
        if (desktopTitle) desktopTitle.value = displayTitle;
        if (mobileTitle) mobileTitle.value = displayTitle;

        const monthIndex = window.calendarState.monthNames.indexOf(schedule.month);
        if (monthIndex !== -1) {
            window.calendarState.current = new Date(schedule.year, monthIndex, 1);
        }

        if (schedule.events) {
            window.calendarState.events = schedule.events;
            localStorage.setItem('calendarEvents_global', JSON.stringify(window.calendarState.events));
        }

        window.calendarState.currentScheduleId = schedule.id;
        window.calendarState.isEditingExisting = true;

        window.renderAll();
        showAlert(`Schedule "${displayTitle}" loaded to editor!`);
    };

    // ✅ Reset Editor
    window.resetEditor = () => {
        window.calendarState.currentScheduleId = null;
        window.calendarState.isEditingExisting = false;
        window.calendarState.events = {};
        localStorage.setItem('calendarEvents_global', JSON.stringify({}));
        
        window.calendarState.current = new Date({{ $year }}, {{ $month - 1 }}, 1);
        
        const month = window.calendarState.monthNames[window.calendarState.current.getMonth()];
        const year = window.calendarState.current.getFullYear();
        const title = `MOD Schedule ${month} ${year}`;
        
        const desktopTitle = document.getElementById('calendar-title');
        const mobileTitle = document.getElementById('calendar-title-mobile');
        
        if (desktopTitle) desktopTitle.value = title;
        if (mobileTitle) mobileTitle.value = title;
        
        window.renderAll();
        showAlert('Editor reset to create new schedule');
    };

    // ✅ Handle window resize
    let resizeTimeout;
    function handleResize() {
        clearTimeout(resizeTimeout);
        resizeTimeout = setTimeout(() => {
            window.renderAll();
        }, 250);
    }

    // ✅ Initialize
    document.addEventListener('DOMContentLoaded', function() {
        window.renderAll();
        window.muatGaleri();

        ['calendar-title', 'calendar-title-mobile'].forEach(id => {
            const el = document.getElementById(id);
            if (el) el.addEventListener('input', (e) => {
                const title = e.target.value;
                document.getElementById('calendar-title-display').textContent = title;
                document.getElementById('calendar-title-display-mobile').textContent = title;
            });
        });

        const deleteAllCheck = document.getElementById('confirmDeleteAllCheck');
        const deleteAllBtn = document.getElementById('confirmDeleteAllBtn');

        if (deleteAllCheck && deleteAllBtn) {
            deleteAllCheck.addEventListener('change', function() {
                deleteAllBtn.disabled = !this.checked;
            });
        }

        const currentMonthDisplay = document.getElementById('currentMonthDisplay');
        if (currentMonthDisplay) {
            const month = window.calendarState.monthNames[window.calendarState.current.getMonth()];
            const year = window.calendarState.current.getFullYear();
            currentMonthDisplay.textContent = `${month} ${year}`;

            let totalEvents = 0;
            Object.values(window.calendarState.events).forEach(events => {
                totalEvents += events.length;
            });
            document.getElementById('totalEventsDisplay').textContent = totalEvents;
        }

        document.getElementById('uploadModal').addEventListener('hidden.bs.modal', function() {
            document.getElementById('modalUploadTitle').textContent = 'Save Schedule to Server';
            document.getElementById('uploadInfoText').textContent = 'Schedule will be saved as an image to the server.';
            document.getElementById('uploadInfoAlert').className = 'alert alert-info';
            document.getElementById('uploadToServerBtn').innerHTML = '<i class="fas fa-cloud-upload-alt me-1"></i> Save to Server';
        });

        window.addEventListener('resize', handleResize);
    });

})();
</script>
@endsection