<?php
namespace App\Http\Controllers;

use App\Models\Cheklist;
use App\Models\Question;
use App\Models\Section;
use App\Models\SubQuestion;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class CheklistController extends Controller
{
    public function index()
    {
        $checklists = Cheklist::oldest()->get();
        $hotels     = $checklists->pluck('hotel')->unique()->values();

        return view('admin.manajemen.checklist', compact('checklists', 'hotels'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'title'    => 'required|string|max:255',
            'hotel'    => 'required|in:HARRIS,POP',
            'mod_type' => 'required|in:general,room_inspection',
        ]);

        Cheklist::create([
            'title'    => $request->title,
            'hotel'    => $request->hotel,
            'mod_type' => $request->mod_type,
        ]);

        return redirect()
            ->route('admin.manajemen.checklist')
            ->with('success', 'Checklist berhasil dibuat.');
    }

    public function update(Request $request, $id)
    {
        $checklist = Cheklist::findOrFail($id);

        $request->validate([
            'title' => 'required|string|max:255',
            'hotel' => 'required|in:HARRIS,POP',
        ]);

        $checklist->update([
            'title' => $request->title,
            'hotel' => $request->hotel,
            // mod_type tidak diupdate saat edit
        ]);

        return redirect()
            ->route('admin.manajemen.checklist')
            ->with('success', 'Checklist berhasil diperbarui.');
    }

    public function destroy($id)
    {
        $checklist = Cheklist::findOrFail($id);
        $checklist->delete();

        return back()->with('success', 'Checklist berhasil dihapus.');
    }

    public function show($id)
    {
        $checklist = Cheklist::findOrFail($id);

        return view('admin.manajemen.checklist-show', compact('checklist'));
    }

    public function sectionindex($id)
    {
        $checklist = Cheklist::findOrFail($id);

        $sections = $checklist->sections()
            ->with(['questions' => function ($query) {
                $query->with(['subQuestions' => function ($q) {
                    $q->orderBy('order');
                }])->orderBy('id');
            }])->orderBy('id')->get();

        return view('admin.manajemen.section', compact('sections', 'checklist'));
    }

    public function sectionstore(Request $request)
    {
        $validated = $request->validate([
            'checklist_id' => 'required|exists:checklists,id',
            'title'        => 'required|string|max:255',
        ]);

        Section::create([
            'checklist_id' => $validated['checklist_id'],
            'title'        => $validated['title'],
        ]);

        return back()->with('success', 'Halaman berhasil ditambahkan.');
    }

    public function sectionupdate(Request $request, $id)
    {
        $section = Section::findOrFail($id);

        $validated = $request->validate([
            'title' => 'required|string|max:255',
        ]);

        $section->update([
            'title' => $validated['title'],
        ]);

        return back()->with('success', 'Halaman berhasil diperbarui.');
    }

    public function sectiondestroy($id)
    {
        $section = Section::findOrFail($id);

        $section->delete();

        return back()->with('success', 'Halaman berhasil dihapus.');
    }

    public function questionedit($id)
    {
        $question  = Question::with('subQuestions')->findOrFail($id);
        $section   = $question->section;
        $checklist = $section->checklist;

        return view('admin.manajemen.question-edit', compact('question', 'section', 'checklist'));
    }

    public function questionupdate(Request $request, $id)
    {
        $question = Question::with('subQuestions')->findOrFail($id);

        // Gunakan tipe asli dari database (karena user tidak boleh ubah tipe)
        $originalType = $question->type;

        // Validasi hanya berdasarkan tipe asli
        $rules = [
            'question_text'        => 'required|string',
            'is_required'          => 'sometimes|boolean',
            'has_comment'          => 'sometimes|boolean',
            'comment_required'     => 'sometimes|boolean',
            'image_required_if_no' => 'sometimes|boolean',
        ];

        // Jika tipe aslinya subQuestionList, validasi sub_questions
        if ($originalType === 'subQuestionList') {
            $rules['sub_questions']                        = 'nullable|array';
            $rules['sub_questions.*.text']                 = 'required|string';
            $rules['sub_questions.*.is_required']          = 'sometimes|boolean';
            $rules['sub_questions.*.image_required_if_no'] = 'sometimes|boolean';
            $rules['sub_questions.*.id']                   = 'sometimes|exists:sub_questions,id,question_id,' . $question->id;
        }

        $validated = $request->validate($rules);

        DB::beginTransaction();
        try {
            // Siapkan data utama untuk update
            $updateData = [
                'question_text'        => $validated['question_text'],
                'is_required'          => $request->boolean('is_required') ? 1 : 0,
                'has_comment'          => $request->boolean('has_comment') ? 1 : 0,
                'comment_required'     => $request->boolean('comment_required') ? 1 : 0,
                'image_required_if_no' => $request->boolean('image_required_if_no') ? 1 : 0,
            ];

            // Atur ulang nilai berdasarkan tipe (karena beberapa tipe non-interaktif)
            if ($originalType === 'yesNo') {
                // Jika has_comment dimatikan, matikan juga comment_required
                if (! $updateData['has_comment']) {
                    $updateData['comment_required'] = 0;
                }
            } elseif (in_array($originalType, ['header', 'readOnlyText', 'pageNote'])) {
                // Non-interaktif: semua flag = 0
                $updateData['is_required']          = 0;
                $updateData['has_comment']          = 0;
                $updateData['comment_required']     = 0;
                $updateData['image_required_if_no'] = 0;
            } elseif ($originalType === 'subQuestionList') {
                // SubQuestionList: pertanyaan utama non-interaktif
                $updateData['is_required']          = 0;
                $updateData['has_comment']          = 0;
                $updateData['comment_required']     = 0;
                $updateData['image_required_if_no'] = 0;

                // Handle sub questions
                $existingSubIds = [];

                if (! empty($validated['sub_questions'])) {
                    foreach ($validated['sub_questions'] as $index => $sub) {
                        if (! empty($sub['id'])) {
                            // Update existing
                            $subQuestion = SubQuestion::where('id', $sub['id'])
                                ->where('question_id', $question->id)
                                ->first();
                            if ($subQuestion) {
                                $subQuestion->update([
                                    'text'                 => trim($sub['text']),
                                    'is_required'          => ($sub['is_required'] ?? false) ? 1 : 0,
                                    'image_required_if_no' => ($sub['image_required_if_no'] ?? false) ? 1 : 0,
                                    'order'                => $index + 1,
                                ]);
                                $existingSubIds[] = $subQuestion->id;
                            }
                        } else {
                            // Create new
                            $newSub = $question->subQuestions()->create([
                                'text'                 => trim($sub['text']),
                                'is_required'          => ($sub['is_required'] ?? false) ? 1 : 0,
                                'image_required_if_no' => ($sub['image_required_if_no'] ?? false) ? 1 : 0,
                                'order'                => $index + 1,
                            ]);
                            $existingSubIds[] = $newSub->id;
                        }
                    }

                    // Hapus yang tidak ada di request
                    $question->subQuestions()
                        ->whereNotIn('id', $existingSubIds)
                        ->delete();
                } else {
                    // Hapus semua sub jika tidak dikirim
                    $question->subQuestions()->delete();
                }
            }

            // Simpan perubahan ke question utama
            $question->update($updateData);

            DB::commit();

            return redirect()->back()->with('success', 'Pertanyaan berhasil diperbarui.');

        } catch (\Exception $e) {
            DB::rollBack();

            return back()->withErrors([
                'error' => 'Gagal memperbarui pertanyaan. Silakan coba lagi.',
            ])->withInput();
        }
    }

    public function questiondestroy($id)
    {
        DB::beginTransaction();

        try {
            $question = Question::with('subQuestions')->findOrFail($id);

            // Ambil section_id untuk reorder nanti
            $sectionId    = $question->checklist_section_id;
            $deletedOrder = $question->order;

            // ✅ Hapus sub questions terlebih dahulu
            if ($question->subQuestions()->count() > 0) {
                $question->subQuestions()->delete();
            }

            // ✅ Hapus question utama
            $question->delete();

            // Reorder questions yang tersisa di section yang sama
            $remainingQuestions = Question::where('checklist_section_id', $sectionId)
                ->where('order', '>', $deletedOrder)
                ->orderBy('order')
                ->get();

            foreach ($remainingQuestions as $q) {
                $q->order = $q->order - 1;
                $q->save();
            }

            DB::commit();

            return redirect()->back()->with('success', 'Pertanyaan berhasil dihapus.');

        } catch (\Exception $e) {
            DB::rollBack();

            error_log('Error deleting question ID ' . $id . ': ' . $e->getMessage());

            return redirect()->back()->withErrors([
                'error' => 'Gagal menghapus pertanyaan: ' . $e->getMessage(),
            ]);
        }
    }
}
