<?php
namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\ChecklistSubmission;
use App\Models\Cheklist;
use App\Models\ModSession;
use App\Models\Question;
use App\Models\QuestionAnswer;
use App\Models\Section;
use App\Models\SubQuestionAnswer;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;

class ChecklistFillController extends Controller
{
    /**
     * DISPLAY FORM PER SECTION
     */
    public function fill($checklistId, $step = 0)
    {
        try {
            // Check if user is logged in
            if (! auth()->check()) {
                return redirect()->route('login')->with('error', 'Please login first');
            }

            // Find checklist
            $checklist = Cheklist::find($checklistId);
            if (! $checklist) {
                return back()->with('error', 'Checklist not found');
            }

            // Get sections
            $sections = Section::where('checklist_id', $checklistId)
                ->orderBy('id')
                ->get();

            if ($sections->isEmpty()) {
                return back()->with('error', 'No sections in this checklist yet');
            }

            // Validate step
            if ($step < 0 || $step >= $sections->count()) {
                $step = 0;
            }

            $currentSection = $sections[$step];
            $user           = auth()->user();
            $today          = now()->toDateString();

            // ===== MAIN FIX: SUBMISSION LOGIC PER ROOM =====
            $submission = null;
            $roomNumber = null;

            if ($checklist->mod_type === 'room_inspection') {
                // For room inspection: get room_number from session
                $roomNumber = session('current_room_number');

                // If room_number exists in session, find submission for this room
                if ($roomNumber) {
                    $submission = ChecklistSubmission::where('checklist_id', $checklist->id)
                        ->where('user_id', $user->id)
                        ->whereDate('submitted_at', $today)
                        ->where('room_number', $roomNumber)
                        ->first();

                    // If no submission for this room, create new (draft)
                    if (! $submission) {
                        $submission               = new ChecklistSubmission();
                        $submission->checklist_id = $checklist->id;
                        $submission->user_id      = $user->id;
                        $submission->submitted_at = now();
                        $submission->room_number  = $roomNumber;
                        $submission->status       = 'draft';
                        $submission->save();
                    }
                } else {
                    // If no room_number in session, redirect to step 0 to input room
                    if ($step > 0) {
                        return redirect()->route('user.checklist.fill', [
                            'checklistId' => $checklistId,
                            'step'        => 0,
                        ])->with('error', 'Please enter room number first in the first section.');
                    }

                    // For step 0 without room_number, create temporary submission (without room_number)
                    $submission = ChecklistSubmission::where('checklist_id', $checklist->id)
                        ->where('user_id', $user->id)
                        ->whereDate('submitted_at', $today)
                        ->whereNull('room_number')
                        ->where('status', 'draft')
                        ->first();

                    if (! $submission) {
                        $submission               = new ChecklistSubmission();
                        $submission->checklist_id = $checklist->id;
                        $submission->user_id      = $user->id;
                        $submission->submitted_at = now();
                        $submission->room_number  = null; // No room_number yet
                        $submission->status       = 'draft';
                        $submission->save();
                    }
                }
            } else {
                // For General Operation (not room inspection)
                $submission = ChecklistSubmission::where('checklist_id', $checklist->id)
                    ->where('user_id', $user->id)
                    ->whereDate('submitted_at', $today)
                    ->first();

                if (! $submission) {
                    $submission               = new ChecklistSubmission();
                    $submission->checklist_id = $checklist->id;
                    $submission->user_id      = $user->id;
                    $submission->submitted_at = now();
                    $submission->status       = 'draft';
                    $submission->save();
                }
            }
            // ===== END FIX =====

            // Get questions for this section
            $questions = Question::where('checklist_section_id', $currentSection->id)
                ->with(['subQuestions' => function ($query) {
                    $query->orderBy('order');
                }])
                ->orderBy('order')
                ->get();

            // Get existing answers
            $questionAnswers = QuestionAnswer::where('checklist_submission_id', $submission->id)
                ->whereIn('question_id', $questions->pluck('id'))
                ->get()
                ->keyBy('question_id');

            // Get sub question answers
            $subQuestionIds     = $questions->pluck('subQuestions.*.id')->flatten();
            $subQuestionAnswers = SubQuestionAnswer::where('checklist_submission_id', $submission->id)
                ->whereIn('sub_question_id', $subQuestionIds)
                ->get()
                ->keyBy('sub_question_id');

            // Get room count for Room Inspection
            $roomCount = 0;
            if ($checklist->mod_type === 'room_inspection') {
                $modSession = ModSession::where('user_id', $user->id)
                    ->where('event_date', $today)
                    ->where('hotel', $checklist->hotel)
                    ->first();
                $roomCount = $modSession ? $modSession->room_inspection_count : 0;
            }

            return view('user.mod.fill', [
                'checklist'          => $checklist,
                'sections'           => $sections,
                'currentSection'     => $currentSection,
                'questions'          => $questions,
                'step'               => $step,
                'totalSteps'         => $sections->count(),
                'submission'         => $submission,
                'questionAnswers'    => $questionAnswers,
                'subQuestionAnswers' => $subQuestionAnswers,
                'roomCount'          => $roomCount,
                'roomNumber'         => $roomNumber,
            ]);

        } catch (\Exception $e) {
            \Log::error('Error in fill(): ' . $e->getMessage() . ' - ' . $e->getTraceAsString());
            return back()->with('error', 'An error occurred: ' . $e->getMessage());
        }
    }

    public function save(Request $request, $checklistId, $step)
    {
        try {
            DB::beginTransaction();

            // Validate input
            $checklist = Cheklist::findOrFail($checklistId);
            $sections  = Section::where('checklist_id', $checklistId)
                ->orderBy('id')
                ->get();

            if (! isset($sections[$step])) {
                throw new \Exception("Step {$step} is invalid");
            }

            $currentSection = $sections[$step];
            $user           = auth()->user();
            $today          = now()->toDateString();

            // ===== IMPROVED SUBMISSION LOGIC WITH BETTER ERROR HANDLING =====
            $submission = null;
            $roomNumber = null;

            if ($checklist->mod_type === 'room_inspection') {
                // Step 0: get room_number from request
                if ($step == 0) {
                    $roomNumber = $request->input('room_number');
                    if (! $roomNumber) {
                        DB::rollBack();
                        return back()->with('error', 'Room number is required');
                    }

                    // Validate: check if room_number already used today (submitted)
                    $roomExistsSubmitted = ChecklistSubmission::where('checklist_id', $checklist->id)
                        ->where('user_id', $user->id)
                        ->whereDate('submitted_at', $today)
                        ->where('room_number', $roomNumber)
                        ->where('status', 'submitted')
                        ->exists();

                    if ($roomExistsSubmitted) {
                        DB::rollBack();
                        return back()
                            ->withInput()
                            ->with('error', 'Room number ' . $roomNumber . ' has already been used for inspection today.');
                    }

                    // Save to session IMMEDIATELY
                    session(['current_room_number' => $roomNumber]);

                    // Check if draft already exists for this room
                    $existingDraft = ChecklistSubmission::where('checklist_id', $checklist->id)
                        ->where('user_id', $user->id)
                        ->whereDate('submitted_at', $today)
                        ->where('room_number', $roomNumber)
                        ->whereIn('status', ['draft', 'in_progress'])
                        ->first();

                    if ($existingDraft) {
                        // Use existing draft
                        $submission = $existingDraft;
                    } else {
                        // Check for other drafts for this checklist today
                        $otherDrafts = ChecklistSubmission::where('checklist_id', $checklist->id)
                            ->where('user_id', $user->id)
                            ->whereDate('submitted_at', $today)
                            ->whereIn('status', ['draft', 'in_progress'])
                            ->get();

                        // Delete any other drafts to prevent duplicates
                        foreach ($otherDrafts as $draft) {
                            try {
                                // Delete related answers first
                                QuestionAnswer::where('checklist_submission_id', $draft->id)->delete();
                                SubQuestionAnswer::where('checklist_submission_id', $draft->id)->delete();
                                // Delete the draft
                                $draft->delete();
                            } catch (\Exception $e) {
                                \Log::warning('Error deleting draft ' . $draft->id . ': ' . $e->getMessage());
                            }
                        }

                        // Create new submission - use firstOrCreate instead of updateOrCreate
                        try {
                            $submission = ChecklistSubmission::firstOrCreate(
                                [
                                    'checklist_id' => $checklist->id,
                                    'user_id'      => $user->id,
                                    'room_number'  => $roomNumber,
                                    'submitted_at' => $today,
                                ],
                                [
                                    'submitted_at' => now(),
                                    'status'       => 'draft',
                                ]
                            );
                        } catch (\Illuminate\Database\QueryException $e) {
                            // If duplicate entry error, try to find existing one
                            if ($e->getCode() == 23000 && strpos($e->getMessage(), 'Duplicate entry') !== false) {
                                // Try to find the existing record
                                $submission = ChecklistSubmission::where('checklist_id', $checklist->id)
                                    ->where('user_id', $user->id)
                                    ->whereDate('submitted_at', $today)
                                    ->where('room_number', $roomNumber)
                                    ->whereIn('status', ['draft', 'in_progress'])
                                    ->first();

                                if (! $submission) {
                                    // If still not found, try without status filter
                                    $submission = ChecklistSubmission::where('checklist_id', $checklist->id)
                                        ->where('user_id', $user->id)
                                        ->whereDate('submitted_at', $today)
                                        ->where('room_number', $roomNumber)
                                        ->first();

                                    if (! $submission) {
                                        // Create a new one with a unique room number
                                        $newRoomNumber = $roomNumber . '_' . time();
                                        $submission    = new ChecklistSubmission([
                                            'checklist_id' => $checklist->id,
                                            'user_id'      => $user->id,
                                            'submitted_at' => now(),
                                            'room_number'  => $newRoomNumber,
                                            'status'       => 'draft',
                                        ]);
                                        $submission->save();
                                        session(['current_room_number' => $newRoomNumber]);
                                    }
                                }
                            } else {
                                throw $e;
                            }
                        }
                    }
                } else {
                    // Step > 0: Handle session recovery
                    $roomNumber = session('current_room_number');

                    // If session lost, try to recover from existing draft
                    if (! $roomNumber) {
                        // Look for any draft submission for this checklist today
                        $draftSubmission = ChecklistSubmission::where('checklist_id', $checklist->id)
                            ->where('user_id', $user->id)
                            ->whereDate('submitted_at', $today)
                            ->whereIn('status', ['draft', 'in_progress'])
                            ->orderBy('created_at', 'desc')
                            ->first();

                        if ($draftSubmission && $draftSubmission->room_number) {
                            // Recover room_number from draft
                            $roomNumber = $draftSubmission->room_number;
                            session(['current_room_number' => $roomNumber]);
                        } else {
                            DB::rollBack();
                            return redirect()->route('user.checklist.fill', [
                                'checklistId' => $checklistId,
                                'step'        => 0,
                            ])->with('error', 'Session expired. Please enter room number again.');
                        }
                    }

                    // Find DRAFT or IN_PROGRESS submission for this room
                    $submission = ChecklistSubmission::where('checklist_id', $checklist->id)
                        ->where('user_id', $user->id)
                        ->whereDate('submitted_at', $today)
                        ->where('room_number', $roomNumber)
                        ->whereIn('status', ['draft', 'in_progress'])
                        ->first();

                    if (! $submission) {
                        // Check if room already submitted
                        $submittedCheck = ChecklistSubmission::where('checklist_id', $checklist->id)
                            ->where('user_id', $user->id)
                            ->whereDate('submitted_at', $today)
                            ->where('room_number', $roomNumber)
                            ->where('status', 'submitted')
                            ->exists();

                        if ($submittedCheck) {
                            DB::rollBack();
                            return redirect()->route('user.mod')
                                ->with('error', 'Room ' . $roomNumber . ' has already been inspected. Please start a new room.');
                        } else {
                            // No draft found and not submitted - redirect to step 0
                            DB::rollBack();
                            return redirect()->route('user.checklist.fill', [
                                'checklistId' => $checklistId,
                                'step'        => 0,
                            ])->with('error', 'No active submission found for room ' . $roomNumber . '. Please start from the beginning.');
                        }
                    }
                }
            } else {
                // For General Operation
                // Find DRAFT or IN_PROGRESS submission for today
                $submission = ChecklistSubmission::where('checklist_id', $checklist->id)
                    ->where('user_id', $user->id)
                    ->whereDate('submitted_at', $today)
                    ->whereNull('room_number')
                    ->whereIn('status', ['draft', 'in_progress'])
                    ->first();

                if (! $submission) {
                    // Check if already submitted
                    $submittedCheck = ChecklistSubmission::where('checklist_id', $checklist->id)
                        ->where('user_id', $user->id)
                        ->whereDate('submitted_at', $today)
                        ->whereNull('room_number')
                        ->where('status', 'submitted')
                        ->exists();

                    if ($submittedCheck) {
                        DB::rollBack();
                        return redirect()->route('user.mod')
                            ->with('info', 'General checklist already completed for today.');
                    }

                    // For step 0, create new draft
                    if ($step == 0) {
                        try {
                            $submission = ChecklistSubmission::firstOrCreate(
                                [
                                    'checklist_id' => $checklist->id,
                                    'user_id'      => $user->id,
                                    'submitted_at' => $today,
                                    'room_number'  => null,
                                ],
                                [
                                    'submitted_at' => now(),
                                    'status'       => 'draft',
                                ]
                            );
                        } catch (\Illuminate\Database\QueryException $e) {
                            // If duplicate, find existing
                            if ($e->getCode() == 23000) {
                                $submission = ChecklistSubmission::where('checklist_id', $checklist->id)
                                    ->where('user_id', $user->id)
                                    ->whereDate('submitted_at', $today)
                                    ->whereNull('room_number')
                                    ->first();

                                if (! $submission) {
                                    // Create new with timestamp to make it unique
                                    $submission = new ChecklistSubmission([
                                        'checklist_id' => $checklist->id,
                                        'user_id'      => $user->id,
                                        'submitted_at' => now(),
                                        'status'       => 'draft',
                                    ]);
                                    $submission->save();
                                }
                            } else {
                                throw $e;
                            }
                        }
                    } else {
                        // For step > 0 without draft, redirect to step 0
                        DB::rollBack();
                        return redirect()->route('user.checklist.fill', [
                            'checklistId' => $checklistId,
                            'step'        => 0,
                        ])->with('error', 'No draft submission found. Please start from the beginning.');
                    }
                }
            }
            // ===== END ENHANCED LOGIC =====

            // Make sure we have a submission
            if (! $submission) {
                DB::rollBack();
                throw new \Exception("No submission found or created for the checklist.");
            }

            // Update status to in_progress if still draft
            if ($submission->status === 'draft') {
                $submission->update(['status' => 'in_progress']);
            }

            // Get questions for validation
            $questions = Question::where('checklist_section_id', $currentSection->id)
                ->with('subQuestions')
                ->orderBy('order')
                ->get();

            // STRICT validation rules (remain the same)
            $rules            = [];
            $messages         = [];
            $customAttributes = [];

            foreach ($questions as $question) {
                // Skip validation for non-interactive types
                if (in_array($question->type, ['header', 'readOnlyText', 'pageNote'])) {
                    continue;
                }

                $customAttributes["question.{$question->id}.answer_yes_no"] = $question->question_text;
                $customAttributes["question.{$question->id}.comment"]       = 'Comment for ' . $question->question_text;
                $customAttributes["question.{$question->id}.image"]         = 'Image for ' . $question->question_text;

                if ($question->type === 'yesNo') {
                    if ($question->is_required) {
                        $rules["question.{$question->id}.answer_yes_no"]             = ['required', 'in:yes,no,good,bad'];
                        $messages["question.{$question->id}.answer_yes_no.required"] = 'Question ":attribute" is required';
                    }

                    if ($question->has_comment) {
                        $rules["question.{$question->id}.comment"]             = ['required', 'string', 'min:3'];
                        $messages["question.{$question->id}.comment.required"] = 'Comment for ":attribute" is required';
                    }

                    $answerValue = $request->input("question.{$question->id}.answer_yes_no");
                    $badValues   = $checklist->mod_type === 'room_inspection' ? ['bad'] : ['no'];

                    if ($question->image_required_if_no && in_array($answerValue, $badValues)) {
                        $rules["question.{$question->id}.image"] = [
                            Rule::requiredIf(function () use ($request, $question) {
                                $existingImage = $request->input("question.{$question->id}.existing_image");
                                return ! $request->hasFile("question.{$question->id}.image") && empty($existingImage);
                            }),
                            'image',
                            'max:5120',
                            'mimes:jpg,jpeg,png,gif,bmp',
                        ];
                        $messages["question.{$question->id}.image.required"] = 'Photo is required if answer for ":attribute" is ' . ($checklist->mod_type === 'room_inspection' ? 'Bad' : 'No');
                    }
                }

                if ($question->type === 'imageUpload') {
                    if ($question->is_required) {
                        $rules["question.{$question->id}.image"] = [
                            Rule::requiredIf(function () use ($request, $question) {
                                $existingImage = $request->input("question.{$question->id}.existing_image");
                                return ! $request->hasFile("question.{$question->id}.image") && empty($existingImage);
                            }),
                            'image',
                            'max:5120',
                            'mimes:jpg,jpeg,png,gif,bmp',
                        ];
                    }

                    if ($question->has_comment) {
                        $rules["question.{$question->id}.comment"]             = ['required', 'string', 'min:3'];
                        $messages["question.{$question->id}.comment.required"] = 'Comment for ":attribute" is required';
                    }
                }

                if ($question->type === 'subQuestionList') {
                    if ($question->has_comment) {
                        $rules["question.{$question->id}.comment"]             = ['required', 'string', 'min:3'];
                        $messages["question.{$question->id}.comment.required"] = 'Comment for ":attribute" is required';
                    }

                    foreach ($question->subQuestions as $subQuestion) {
                        $customAttributes["subquestion.{$subQuestion->id}.answer_yes_no"] = $subQuestion->text;
                        $customAttributes["subquestion.{$subQuestion->id}.comment"]       = 'Comment for ' . $subQuestion->text;
                        $customAttributes["subquestion.{$subQuestion->id}.image"]         = 'Image for ' . $subQuestion->text;

                        if ($subQuestion->is_required) {
                            $rules["subquestion.{$subQuestion->id}.answer_yes_no"]             = ['required', 'in:yes,no,good,bad'];
                            $messages["subquestion.{$subQuestion->id}.answer_yes_no.required"] = 'Sub question ":attribute" is required';

                            $rules["subquestion.{$subQuestion->id}.comment"]             = ['required', 'string', 'min:3'];
                            $messages["subquestion.{$subQuestion->id}.comment.required"] = 'Comment for ":attribute" is required';
                        }

                        $subAnswerValue = $request->input("subquestion.{$subQuestion->id}.answer_yes_no");
                        $badValues      = $checklist->mod_type === 'room_inspection' ? ['bad'] : ['no'];

                        if ($subQuestion->image_required_if_no && in_array($subAnswerValue, $badValues)) {
                            $rules["subquestion.{$subQuestion->id}.image"] = [
                                Rule::requiredIf(function () use ($request, $subQuestion) {
                                    $existingImage = $request->input("subquestion.{$subQuestion->id}.existing_image");
                                    return ! $request->hasFile("subquestion.{$subQuestion->id}.image") && empty($existingImage);
                                }),
                                'image',
                                'max:5120',
                                'mimes:jpg,jpeg,png,gif,bmp',
                            ];
                        }
                    }
                }
            }

            // Validate overall comment for room inspection at last step
            $isLastStep = ($step + 1) >= $sections->count();
            if ($isLastStep && $checklist->mod_type === 'room_inspection') {
                $rules['overall_comment']             = ['required', 'string', 'min:10'];
                $messages['overall_comment.required'] = 'Overall comment is required';
                $messages['overall_comment.min']      = 'Overall comment must be at least 10 characters';
            }

            // Run validation
            $validated = $request->validate($rules, $messages, $customAttributes);

            // Save data
            foreach ($questions as $question) {
                if (in_array($question->type, ['header', 'readOnlyText', 'pageNote'])) {
                    continue;
                }

                $qInput         = $request->input("question.{$question->id}", []);
                $existingAnswer = QuestionAnswer::where('checklist_submission_id', $submission->id)
                    ->where('question_id', $question->id)
                    ->first();

                $imagePath = $existingAnswer?->image_path;

                if ($request->hasFile("question.{$question->id}.image")) {
                    if ($imagePath && Storage::disk('public')->exists($imagePath)) {
                        Storage::disk('public')->delete($imagePath);
                    }
                    $imagePath = $request->file("question.{$question->id}.image")
                        ->store('checklist_images', 'public');
                } elseif (! empty($qInput['existing_image'])) {
                    $imagePath = $qInput['existing_image'];
                }

                $answerData = [
                    'room_number'   => $checklist->mod_type === 'room_inspection' ? $roomNumber : null,
                    'answer_yes_no' => $qInput['answer_yes_no'] ?? null,
                    'comment'       => $qInput['comment'] ?? null,
                    'image_path'    => $imagePath,
                ];

                if (empty($answerData['comment'])) {
                    $answerData['comment'] = null;
                }

                QuestionAnswer::updateOrCreate(
                    [
                        'checklist_submission_id' => $submission->id,
                        'question_id'             => $question->id,
                    ],
                    $answerData
                );

                // Handle sub questions
                if ($question->type === 'subQuestionList') {
                    foreach ($question->subQuestions as $subQuestion) {
                        $subInput          = $request->input("subquestion.{$subQuestion->id}", []);
                        $existingSubAnswer = SubQuestionAnswer::where('checklist_submission_id', $submission->id)
                            ->where('sub_question_id', $subQuestion->id)
                            ->first();

                        $subImagePath = $existingSubAnswer?->image_path;

                        if ($request->hasFile("subquestion.{$subQuestion->id}.image")) {
                            if ($subImagePath && Storage::disk('public')->exists($subImagePath)) {
                                Storage::disk('public')->delete($subImagePath);
                            }
                            $subImagePath = $request->file("subquestion.{$subQuestion->id}.image")
                                ->store('checklist_images', 'public');
                        } elseif (! empty($subInput['existing_image'])) {
                            $subImagePath = $subInput['existing_image'];
                        }

                        $subAnswerData = [
                            'room_number'   => $checklist->mod_type === 'room_inspection' ? $roomNumber : null,
                            'answer_yes_no' => $subInput['answer_yes_no'] ?? null,
                            'comment'       => $subInput['comment'] ?? null,
                            'image_path'    => $subImagePath,
                        ];

                        SubQuestionAnswer::updateOrCreate(
                            [
                                'checklist_submission_id' => $submission->id,
                                'sub_question_id'         => $subQuestion->id,
                            ],
                            $subAnswerData
                        );
                    }
                }
            }

            // Save overall comment
            if ($isLastStep && $checklist->mod_type === 'room_inspection') {
                $submission->update([
                    'overall_comment' => $request->input('overall_comment'),
                ]);
            }

            DB::commit();

            // Determine next step
            if ($isLastStep) {
                // Last section: submit this room
                $submission->update([
                    'status'       => 'submitted',
                    'submitted_at' => now(),
                ]);

                // Update mod_session untuk hotel ini
                $modSession = ModSession::firstOrCreate(
                    [
                        'user_id'    => $user->id,
                        'event_date' => $today,
                        'hotel'      => $checklist->hotel,
                    ],
                    [
                        'status' => 'in_progress',
                        'hotel'  => $checklist->hotel,
                    ]
                );

                if ($checklist->mod_type === 'room_inspection') {
                    $modSession->increment('room_inspection_count');
                    $roomCount = $modSession->fresh()->room_inspection_count;
                    if ($roomCount >= 5) {
                        $modSession->update(['room_inspection_completed' => true]);
                    }

                    // Clear session untuk room inspection
                    session()->forget('current_room_number');
                } else {
                    // Update general_completed untuk hotel ini
                    $modSession->update(['general_completed' => true]);
                }

                // Check if all MOD completed untuk hotel ini
                if ($modSession->general_completed && $modSession->room_inspection_completed) {
                    $modSession->update(['status' => 'completed']);
                }

                // Redirect based on type
                if ($checklist->mod_type === 'room_inspection') {
                    $roomCount = $modSession->room_inspection_count;
                    if ($roomCount < 5) {
                        return redirect()
                            ->route('user.checklist.fill', [
                                'checklistId' => $checklist->id,
                                'step'        => 0,
                            ])
                            ->with('success', "Room {$roomCount} submitted successfully! Please enter data for the next room.");
                    } else {
                        return redirect()
                            ->route('user.mod')
                            ->with('success', 'Room Inspection for ' . $checklist->hotel . ' completed! All 5 rooms have been inspected.');
                    }
                } else {
                    return redirect()
                        ->route('user.mod')
                        ->with('success', $checklist->hotel . ' General Operation checklist submitted successfully!');
                }
            }

            return redirect()->route('user.checklist.fill', [
                'checklistId' => $checklist->id,
                'step'        => $step + 1,
            ])->with('success', 'Data saved successfully');

        } catch (ValidationException $e) {
            DB::rollBack();
            return back()
                ->withInput()
                ->withErrors($e->errors())
                ->with('error', 'Please complete all required fields');
        } catch (\Illuminate\Database\QueryException $e) {
            DB::rollBack();
            \Log::error('Database error in save(): ' . $e->getMessage() . ' - SQL: ' . $e->getSql() . ' - Bindings: ' . json_encode($e->getBindings()));

            // Check for specific duplicate entry error
            if ($e->getCode() == 23000) {
                // Check if it's a duplicate entry error
                if (strpos($e->getMessage(), 'Duplicate entry') !== false) {
                    return back()
                        ->withInput()
                        ->with('error', 'This submission already exists. Please refresh the page and try again.');
                }
            }

            return back()
                ->withInput()
                ->with('error', 'Failed to save to database. Please try again. Error: ' . $e->getMessage());
        } catch (Exception $e) {
            DB::rollBack();
            \Log::error('Exception in save(): ' . $e->getMessage() . ' - ' . $e->getTraceAsString());
            return back()
                ->withInput()
                ->with('error', 'Failed to save: ' . $e->getMessage());
        }
    }

    /**
     * AUTO-SAVE PROGRESS VIA AJAX (no strict validation)
     */
    public function autoSave(Request $request, $checklistId, $step)
    {
        try {
            $user      = auth()->user();
            $today     = now()->toDateString();
            $checklist = Cheklist::findOrFail($checklistId);

            // Get submission
            $submission = null;
            $roomNumber = null;

            if ($checklist->mod_type === 'room_inspection') {
                $roomNumber = session('current_room_number');

                if (! $roomNumber && $step == 0) {
                    $roomNumber = $request->input('room_number');
                    if ($roomNumber) {
                        session(['current_room_number' => $roomNumber]);
                    }
                }

                if ($roomNumber) {
                    $submission = ChecklistSubmission::where('checklist_id', $checklist->id)
                        ->where('user_id', $user->id)
                        ->whereDate('submitted_at', $today)
                        ->where('room_number', $roomNumber)
                        ->first();
                }
            } else {
                $submission = ChecklistSubmission::where('checklist_id', $checklist->id)
                    ->where('user_id', $user->id)
                    ->whereDate('submitted_at', $today)
                    ->first();
            }

            // Create draft if not exists
            if (! $submission) {
                $submission = ChecklistSubmission::create([
                    'checklist_id' => $checklist->id,
                    'user_id'      => $user->id,
                    'submitted_at' => now(),
                    'room_number'  => $roomNumber,
                    'status'       => 'draft',
                ]);
            }

            // Get current section
            $section = Section::where('checklist_id', $checklistId)
                ->orderBy('id')
                ->skip($step)
                ->first();

            if (! $section) {
                return response()->json([
                    'success' => false,
                    'message' => 'Section not found',
                ], 404);
            }

            // Get questions
            $questions = Question::where('checklist_section_id', $section->id)
                ->with('subQuestions')
                ->get();

            // Save answers without strict validation
            foreach ($questions as $question) {
                if (in_array($question->type, ['header', 'readOnlyText', 'pageNote'])) {
                    continue;
                }

                $qInput         = $request->input("question.{$question->id}", []);
                $existingAnswer = QuestionAnswer::where('checklist_submission_id', $submission->id)
                    ->where('question_id', $question->id)
                    ->first();

                // Handle image
                $imagePath = $existingAnswer?->image_path;
                if ($request->hasFile("question.{$question->id}.image")) {
                    if ($imagePath && Storage::disk('public')->exists($imagePath)) {
                        Storage::disk('public')->delete($imagePath);
                    }
                    $imagePath = $request->file("question.{$question->id}.image")
                        ->store('checklist_images', 'public');
                }

                QuestionAnswer::updateOrCreate(
                    [
                        'checklist_submission_id' => $submission->id,
                        'question_id'             => $question->id,
                    ],
                    [
                        'room_number'   => $roomNumber,
                        'answer_yes_no' => $qInput['answer_yes_no'] ?? null,
                        'comment'       => $qInput['comment'] ?? null,
                        'image_path'    => $imagePath,
                    ]
                );

                // Save sub-questions
                if ($question->type === 'subQuestionList') {
                    foreach ($question->subQuestions as $subQuestion) {
                        $subInput          = $request->input("subquestion.{$subQuestion->id}", []);
                        $existingSubAnswer = SubQuestionAnswer::where('checklist_submission_id', $submission->id)
                            ->where('sub_question_id', $subQuestion->id)
                            ->first();

                        $subImagePath = $existingSubAnswer?->image_path;
                        if ($request->hasFile("subquestion.{$subQuestion->id}.image")) {
                            if ($subImagePath && Storage::disk('public')->exists($subImagePath)) {
                                Storage::disk('public')->delete($subImagePath);
                            }
                            $subImagePath = $request->file("subquestion.{$subQuestion->id}.image")
                                ->store('checklist_images', 'public');
                        }

                        SubQuestionAnswer::updateOrCreate(
                            [
                                'checklist_submission_id' => $submission->id,
                                'sub_question_id'         => $subQuestion->id,
                            ],
                            [
                                'room_number'   => $roomNumber,
                                'answer_yes_no' => $subInput['answer_yes_no'] ?? null,
                                'comment'       => $subInput['comment'] ?? null,
                                'image_path'    => $subImagePath,
                            ]
                        );
                    }
                }
            }

            // Save overall comment
            if ($request->filled('overall_comment')) {
                $submission->update(['overall_comment' => $request->input('overall_comment')]);
            }

            return response()->json([
                'success'       => true,
                'message'       => 'Progress saved automatically',
                'last_saved'    => now()->format('h:i A'),
                'submission_id' => $submission->id,
            ]);

        } catch (\Exception $e) {
            \Log::error('Auto-save error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to save progress',
            ], 500);
        }
    }

    /**
     * PREVIEW / REVIEW CHECKLIST
     */
    /**
     * PREVIEW / REVIEW CHECKLIST - FIXED VERSION
     */
/**
 * PREVIEW / REVIEW CHECKLIST - FIXED VERSION
 */
/**
 * PREVIEW / REVIEW CHECKLIST - COMPLETE FIXED VERSION
 */
    public function review($checklistId)
    {
        try {
            // Check if user is logged in
            if (! auth()->check()) {
                return redirect()->route('login')->with('error', 'Please login first');
            }

            $user  = auth()->user();
            $today = now()->toDateString();

            // Find checklist
            $checklist = Cheklist::find($checklistId);
            if (! $checklist) {
                return back()->with('error', 'Checklist not found');
            }

            // Get today's submissions for this checklist
            $submissionsQuery = ChecklistSubmission::where('checklist_id', $checklistId)
                ->where('user_id', $user->id)
                ->whereDate('submitted_at', $today)
                ->where('status', 'submitted');

            // Filter by room number based on checklist type
            if ($checklist->mod_type === 'room_inspection') {
                $submissionsQuery->whereNotNull('room_number');
            } else {
                $submissionsQuery->whereNull('room_number');
            }

            $submissions = $submissionsQuery->orderBy('room_number', 'asc')
                ->orderBy('created_at', 'asc')
                ->get();

            if ($submissions->isEmpty()) {
                return redirect()->route('user.checklist.fill', [
                    'checklistId' => $checklistId,
                    'step'        => 0,
                ])->with('error', 'No completed inspections found for today. Please complete an inspection first.');
            }

            // Group submissions by room number for room inspection
            $groupedSubmissions = [];
            if ($checklist->mod_type === 'room_inspection') {
                foreach ($submissions as $submission) {
                    if ($submission->room_number) {
                        $groupedSubmissions[$submission->room_number] = $submission;
                    }
                }
                ksort($groupedSubmissions);
            } else {
                // For general operation, use the first submission
                $firstSubmission    = $submissions->first();
                $groupedSubmissions = ['General' => $firstSubmission];
            }

            // Get all sections with questions and sub-questions
            $sections = Section::where('checklist_id', $checklistId)
                ->with(['questions' => function ($query) {
                    $query->orderBy('order')
                        ->with(['subQuestions' => function ($q) {
                            $q->orderBy('order');
                        }]);
                }])
                ->orderBy('id')
                ->get();

            // Get all question IDs from these sections
            $questionIds    = $sections->pluck('questions.*.id')->flatten();
            $subQuestionIds = $sections->pluck('questions.*.subQuestions.*.id')->flatten();

            // Preload all answers for all submissions
            $allQuestionAnswers    = [];
            $allSubQuestionAnswers = [];

            foreach ($submissions as $submission) {
                // Load question answers for this submission
                $questionAnswers = QuestionAnswer::where('checklist_submission_id', $submission->id)
                    ->whereIn('question_id', $questionIds)
                    ->get()
                    ->keyBy('question_id');

                $allQuestionAnswers[$submission->id] = $questionAnswers;

                // Load sub-question answers for this submission
                if ($subQuestionIds->isNotEmpty()) {
                    $subQuestionAnswers = SubQuestionAnswer::where('checklist_submission_id', $submission->id)
                        ->whereIn('sub_question_id', $subQuestionIds)
                        ->get()
                        ->keyBy('sub_question_id');

                    $allSubQuestionAnswers[$submission->id] = $subQuestionAnswers;
                } else {
                    $allSubQuestionAnswers[$submission->id] = collect();
                }
            }

            return view('user.mod.review', [
                'checklist'             => $checklist,
                'submissions'           => $groupedSubmissions,
                'sections'              => $sections,
                'allQuestionAnswers'    => $allQuestionAnswers,
                'allSubQuestionAnswers' => $allSubQuestionAnswers,
            ]);

        } catch (\Exception $e) {
            \Log::error('Error in review(): ' . $e->getMessage() . ' - ' . $e->getTraceAsString());
            return back()->with('error', 'An error occurred while loading the review: ' . $e->getMessage());
        }
    }
    /**
     * RESUME / EDIT SUBMISSION
     */
    public function resume($submissionId, $step = 0)
    {
        $submission = ChecklistSubmission::with(['checklist'])
            ->findOrFail($submissionId);

        // Check ownership
        if ($submission->user_id !== auth()->id()) {
            abort(403);
        }

        $checklist = $submission->checklist;
        $sections  = Section::where('checklist_id', $checklist->id)
            ->orderBy('id')
            ->get();

        if (! isset($sections[$step])) {
            abort(404);
        }

        $currentSection = $sections[$step];
        $questions      = Question::where('checklist_section_id', $currentSection->id)
            ->with('subQuestions')
            ->orderBy('order')
            ->get();

        $questionAnswers = QuestionAnswer::where('checklist_submission_id', $submission->id)
            ->whereIn('question_id', $questions->pluck('id'))
            ->get()
            ->keyBy('question_id');

        $subQuestionIds     = $questions->pluck('subQuestions.*.id')->flatten();
        $subQuestionAnswers = SubQuestionAnswer::where('checklist_submission_id', $submission->id)
            ->whereIn('sub_question_id', $subQuestionIds)
            ->get()
            ->keyBy('sub_question_id');

        $roomCount = 0;
        if ($checklist->mod_type === 'room_inspection') {
            $modSession = ModSession::where('user_id', auth()->id())
                ->where('event_date', now()->toDateString())
                ->where('hotel', $checklist->hotel)
                ->first();
            $roomCount = $modSession ? $modSession->room_inspection_count : 0;
        }

        $roomNumber = null;
        if ($checklist->mod_type === 'room_inspection' && $submission->room_number) {
            $roomNumber = $submission->room_number;
            session(['current_room_number' => $roomNumber]);
        }

        return view('user.mod.fill', [
            'checklist'          => $checklist,
            'sections'           => $sections,
            'currentSection'     => $currentSection,
            'questions'          => $questions,
            'step'               => $step,
            'totalSteps'         => $sections->count(),
            'submission'         => $submission,
            'questionAnswers'    => $questionAnswers,
            'subQuestionAnswers' => $subQuestionAnswers,
            'roomCount'          => $roomCount,
            'roomNumber'         => $roomNumber,
            'isResume'           => true,
        ]);
    }

    /**
     * DELETE IMAGE
     */
    public function deleteImage(Request $request)
    {
        $request->validate([
            'type' => 'required|in:question,subquestion',
            'id'   => 'required|integer',
        ]);

        try {
            DB::beginTransaction();

            if ($request->type === 'question') {
                $answer = QuestionAnswer::findOrFail($request->id);
                if ($answer->checklistSubmission->user_id !== auth()->id()) {
                    return response()->json(['success' => false, 'message' => 'Unauthorized'], 403);
                }

                if ($answer->image_path && Storage::disk('public')->exists($answer->image_path)) {
                    Storage::disk('public')->delete($answer->image_path);
                }

                $answer->update(['image_path' => null]);
            } elseif ($request->type === 'subquestion') {
                $answer = SubQuestionAnswer::findOrFail($request->id);
                if ($answer->checklistSubmission->user_id !== auth()->id()) {
                    return response()->json(['success' => false, 'message' => 'Unauthorized'], 403);
                }

                if ($answer->image_path && Storage::disk('public')->exists($answer->image_path)) {
                    Storage::disk('public')->delete($answer->image_path);
                }

                $answer->update(['image_path' => null]);
            }

            DB::commit();
            return response()->json(['success' => true]);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Error deleting image: ' . $e->getMessage());
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    /**
     * RESET PROGRESS
     */
    public function resetProgress(Request $request, $submissionId)
    {
        try {
            $submission = ChecklistSubmission::where('id', $submissionId)
                ->where('user_id', auth()->id())
                ->where('status', 'draft')
                ->firstOrFail();

            // Delete all related answers
            QuestionAnswer::where('checklist_submission_id', $submission->id)->delete();
            SubQuestionAnswer::where('checklist_submission_id', $submission->id)->delete();

            // Delete session room_number if exists
            session()->forget('current_room_number');

            // Delete submission
            $submission->delete();

            return response()->json([
                'success' => true,
                'message' => 'Progress reset successfully',
            ]);

        } catch (\Exception $e) {
            \Log::error('Error resetting progress: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to reset progress. Please try again.',
            ], 500);
        }
    }

    /**
     * VALIDATE ROOM NUMBER (AJAX)
     */
    public function validateRoomNumber(Request $request)
    {
        $request->validate([
            'room_number' => 'required|string|max:20',
        ]);

        try {
            $exists = ChecklistSubmission::where('room_number', $request->room_number)
                ->where('user_id', auth()->id())
                ->whereDate('submitted_at', now()->toDateString())
                ->where('status', 'submitted') // Only check submitted ones
                ->exists();

            return response()->json([
                'success' => true,
                'exists'  => $exists,
                'message' => $exists ? 'Room number already used today' : 'Room number available',
            ]);

        } catch (\Exception $e) {
            \Log::error('Error validating room number: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Validation error occurred',
            ], 500);
        }
    }

    public function report($checklistId)
{
    try {
        // Validasi login
        if (!auth()->check()) {
            return redirect()->route('login')->with('error', 'Silakan login terlebih dahulu');
        }

        $user = auth()->user();
        $today = now()->toDateString();

        // Cari checklist
        $checklist = Cheklist::find($checklistId);
        if (!$checklist) {
            return back()->with('error', 'Checklist tidak ditemukan');
        }

        // Ambil semua submission yang sudah disubmit hari ini
        $submissionsQuery = ChecklistSubmission::where('checklist_id', $checklistId)
            ->where('user_id', $user->id)
            ->whereDate('submitted_at', $today)
            ->where('status', 'submitted');

        if ($checklist->mod_type === 'room_inspection') {
            $submissionsQuery->whereNotNull('room_number');
        } else {
            $submissionsQuery->whereNull('room_number');
        }

        $submissions = $submissionsQuery->orderBy('room_number', 'asc')
            ->orderBy('created_at', 'asc')
            ->get();

        if ($submissions->isEmpty()) {
            return redirect()->route('user.checklist.fill', [
                'checklistId' => $checklistId,
                'step' => 0,
            ])->with('error', 'Belum ada inspeksi yang diselesaikan hari ini');
        }

        // Group submissions untuk room inspection
        $groupedSubmissions = [];
        if ($checklist->mod_type === 'room_inspection') {
            foreach ($submissions as $submission) {
                if ($submission->room_number) {
                    $groupedSubmissions[$submission->room_number] = $submission;
                }
            }
            ksort($groupedSubmissions);
        } else {
            $groupedSubmissions = ['General' => $submissions->first()];
        }

        // Ambil sections & questions
        $sections = Section::where('checklist_id', $checklistId)
            ->with(['questions' => function ($q) {
                $q->orderBy('order')->with('subQuestions');
            }])
            ->orderBy('id')
            ->get();

        // Ambil semua jawaban untuk semua submission
        $allQuestionAnswers = [];
        $allSubQuestionAnswers = [];

        foreach ($submissions as $submission) {
            $questionAnswers = QuestionAnswer::where('checklist_submission_id', $submission->id)
                ->get()
                ->keyBy('question_id');
            $allQuestionAnswers[$submission->id] = $questionAnswers;

            $subQuestionIds = $sections->pluck('questions.*.subQuestions.*.id')->flatten();
            if ($subQuestionIds->isNotEmpty()) {
                $subQuestionAnswers = SubQuestionAnswer::where('checklist_submission_id', $submission->id)
                    ->whereIn('sub_question_id', $subQuestionIds)
                    ->get()
                    ->keyBy('sub_question_id');
                $allSubQuestionAnswers[$submission->id] = $subQuestionAnswers;
            } else {
                $allSubQuestionAnswers[$submission->id] = collect();
            }
        }

        // Ambil MOD session
        $modSession = ModSession::where('user_id', $user->id)
            ->where('event_date', $today)
            ->where('hotel', $checklist->hotel)
            ->first();

        return view('user.mod.report', [
            'checklist' => $checklist,
            'submissions' => $groupedSubmissions,
            'sections' => $sections,
            'allQuestionAnswers' => $allQuestionAnswers,
            'allSubQuestionAnswers' => $allSubQuestionAnswers,
            'modSession' => $modSession,
        ]);

    } catch (\Exception $e) {
        \Log::error('Error in report(): ' . $e->getMessage() . ' - ' . $e->getTraceAsString());
        return back()->with('error', 'Gagal memuat laporan: ' . $e->getMessage());
    }
}
}
